package com.abloq.purix.viewmodel

import android.app.Application
import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import androidx.core.content.ContextCompat
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.session.MediaController
import androidx.media3.session.SessionToken
import android.content.ComponentName
import com.google.common.util.concurrent.ListenableFuture
import com.abloq.purix.models.Category
import com.abloq.purix.models.Song
import com.abloq.purix.service.PlaybackService
import com.abloq.purix.utils.ManifestLoader
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch

class MediaViewModel(application: Application) : AndroidViewModel(application) {

    private var player: Player? = null
    private var controllerFuture: ListenableFuture<MediaController>? = null
    
    private val _currentCategory = MutableLiveData<Category?>()
    val currentCategory: LiveData<Category?> = _currentCategory

    private val _songs = MutableLiveData<List<Song>>(emptyList())
    val songs: LiveData<List<Song>> = _songs

    private val _currentSong = MutableLiveData<Song?>()
    val currentSong: LiveData<Song?> = _currentSong

    private val _isPlaying = MutableLiveData(false)
    val isPlaying: LiveData<Boolean> = _isPlaying

    private val _position = MutableLiveData<Long>(0L)
    val position: LiveData<Long> = _position

    private val _duration = MutableLiveData<Long>(0L)
    val duration: LiveData<Long> = _duration

    private val _artworkData = MutableLiveData<ByteArray?>(null)
    val artworkData: LiveData<ByteArray?> = _artworkData

    private val _queue = MutableLiveData<MutableList<Song>>(mutableListOf())
    val queue: LiveData<MutableList<Song>> = _queue

    private val _nextSong = MutableLiveData<Song?>(null)
    val nextSong: LiveData<Song?> = _nextSong

    private val _volume = MutableLiveData<Float>(1f)
    val volume: LiveData<Float> = _volume

    private val prefs = application.getSharedPreferences("PuriXPrefs", Context.MODE_PRIVATE)
    
    private var seekSpeed = 5000L // Default 5 seconds per tick

    private val handler = Handler(Looper.getMainLooper())
    private val updatePositionRunnable = object : Runnable {
        override fun run() {
            _position.value = player?.currentPosition ?: 0L
            _duration.value = player?.duration?.coerceAtLeast(0L) ?: 0L
            handler.postDelayed(this, 500)
        }
    }
    
    private var seekJob: kotlinx.coroutines.Job? = null

    init {
        val sessionToken = SessionToken(application, ComponentName(application, PlaybackService::class.java))
        controllerFuture = MediaController.Builder(application, sessionToken).buildAsync()
        controllerFuture?.addListener({
            player = controllerFuture?.get()
            setupPlayer()
        }, ContextCompat.getMainExecutor(application))
    }

    private fun setupPlayer() {
        player?.addListener(object : Player.Listener {
            override fun onIsPlayingChanged(isPlaying: Boolean) {
                _isPlaying.value = isPlaying
                if (isPlaying) {
                    handler.post(updatePositionRunnable)
                } else {
                    handler.removeCallbacks(updatePositionRunnable)
                }
            }

            override fun onMediaItemTransition(mediaItem: MediaItem?, reason: Int) {
                val index = player?.currentMediaItemIndex ?: -1
                if (index != -1 && index < (_songs.value?.size ?: 0)) {
                    _currentSong.value = _songs.value?.get(index)
                    
                    val nextIndex = index + 1
                    if (nextIndex < (_songs.value?.size ?: 0)) {
                        _nextSong.value = _songs.value?.get(nextIndex)
                    } else {
                        _nextSong.value = null
                    }
                }
            }
            override fun onMediaMetadataChanged(mediaMetadata: androidx.media3.common.MediaMetadata) {
                _artworkData.value = mediaMetadata.artworkData
            }
        })
        loadSettings() // Defer loadSettings until player is ready
    }

    private fun loadSettings() {
        val savedCatName = prefs.getString("selected_category_name", null)
        val savedCatPath = prefs.getString("selected_category_path", null)
        if (savedCatName != null && savedCatPath != null) {
            selectCategory(Category(savedCatName, savedCatPath))
        }
        seekSpeed = prefs.getLong("seek_speed", 5000L)
    }

    fun selectCategory(category: Category?) {
        _currentCategory.value = category
        if (category == null) {
            prefs.edit().apply {
                remove("selected_category_name")
                remove("selected_category_path")
                apply()
            }
            player?.stop()
            player?.clearMediaItems()
            _songs.value = emptyList()
            _currentSong.value = null
            return
        }

        prefs.edit().apply {
            putString("selected_category_name", category.name)
            putString("selected_category_path", category.path)
            apply()
        }
        
        val loadedSongs = ManifestLoader.loadCategoryManifest(getApplication(), category.path)
        val banned = prefs.getStringSet("banned_songs", emptySet()) ?: emptySet()
        val filteredSongs = loadedSongs.filter { !banned.contains(it.path) }
        _songs.value = filteredSongs
        
        player?.clearMediaItems()
        val mediaItems = filteredSongs.map { song ->
            MediaItem.Builder()
                .setUri("asset:///${song.path}")
                .setMediaId(song.path)
                .build()
        }
        player?.addMediaItems(mediaItems)
        player?.prepare()
    }
    
    fun clearCategory() {
        selectCategory(null)
    }

    fun playSong(song: Song, clearQueue: Boolean = false) {
        val index = _songs.value?.indexOf(song) ?: -1
        if (index != -1) {
            if (clearQueue) {
                // To keep it simple mapping to loaded songs logic
                _queue.value = mutableListOf()
                val loadedSongs = _songs.value ?: emptyList()
                player?.clearMediaItems()
                val mediaItems = loadedSongs.map { s ->
                    MediaItem.Builder()
                        .setUri("asset:///${s.path}")
                        .setMediaId(s.path)
                        .build()
                }
                player?.addMediaItems(mediaItems)
                player?.seekTo(index, 0L)
            } else {
                player?.seekTo(index, 0L)
            }
            player?.play()
        }
    }

    fun togglePlayPause() {
        val p = player ?: return
        if (p.isPlaying) {
            p.pause()
        } else {
            p.play()
        }
    }

    fun skipNext() {
        if (player?.hasNextMediaItem() == true) {
            player?.seekToNext()
        }
    }

    fun skipPrevious() {
        if (player?.hasPreviousMediaItem() == true) {
            player?.seekToPrevious()
        }
    }

    fun seekTo(position: Long) {
        player?.seekTo(position)
    }

    fun setVolume(volume: Float) {
        player?.volume = volume
        _volume.value = volume
    }

    fun startSeeking(forward: Boolean) {
        val currentJob = seekJob
        seekJob?.cancel()
        seekJob = viewModelScope.launch {
            while (isActive) {
                val current = player?.currentPosition ?: 0L
                val target = if (forward) current + seekSpeed else current - seekSpeed
                player?.seekTo(target.coerceIn(0, player?.duration ?: 0L))
                delay(200)
            }
        }
    }

    fun stopSeeking() {
        seekJob?.cancel()
        seekJob = null
    }

    fun addToQueue(song: Song) {
        val currentQueue = _queue.value ?: mutableListOf()
        currentQueue.add(song)
        _queue.value = currentQueue
        
        val mediaItem = MediaItem.Builder()
            .setUri("asset:///${song.path}")
            .setMediaId(song.path)
            .build()
            
        val currentIndex = player?.currentMediaItemIndex ?: -1
        val insertIndex = if (currentIndex != -1) currentIndex + 1 else (player?.mediaItemCount ?: 0)
        player?.addMediaItem(insertIndex, mediaItem)
        
        // Update live _songs to match new playlist
        val currentList = _songs.value?.toMutableList() ?: mutableListOf()
        if (insertIndex <= currentList.size) {
            currentList.add(insertIndex, song)
            _songs.value = currentList
        }
    }

    fun banCurrentSong() {
        val song = _currentSong.value ?: return
        
        // Save to SharedPreferences
        val banned = prefs.getStringSet("banned_songs", mutableSetOf())?.toMutableSet() ?: mutableSetOf()
        banned.add(song.path)
        prefs.edit().putStringSet("banned_songs", banned).apply()
        
        // Remove from current playing queue
        val index = _songs.value?.indexOf(song) ?: -1
        if (index != -1) {
            val currentList = _songs.value?.toMutableList() ?: mutableListOf()
            currentList.removeAt(index)
            _songs.value = currentList
            
            player?.removeMediaItem(index)
            
            if (currentList.isEmpty()) {
                player?.stop()
                _currentSong.value = null
            }
        }
    }

    fun updateSeekSpeed(speedMs: Long) {
        seekSpeed = speedMs
        prefs.edit().putLong("seek_speed", speedMs).apply()
    }

    override fun onCleared() {
        super.onCleared()
        handler.removeCallbacksAndMessages(null)
        controllerFuture?.let { 
            MediaController.releaseFuture(it) 
        }
    }
}
