package com.abloq.purix.ui.screens

import androidx.compose.foundation.layout.*
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowBack
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import com.abloq.purix.R
import com.abloq.purix.viewmodel.MediaViewModel
import com.abloq.purix.ui.components.AboutDialog

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SettingsScreen(
    viewModel: MediaViewModel,
    onBack: () -> Unit,
    onResetCategory: () -> Unit
) {
    var showKosherDialog by remember { mutableStateOf(false) }
    var showAbout by remember { mutableStateOf(false) }
    var seekSpeed by remember { mutableStateOf(5000f) } // Default 5s

    if (showAbout) {
        AboutDialog(onDismiss = { showAbout = false })
    }

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text(stringResource(R.string.settings), fontWeight = FontWeight.Bold) },
                navigationIcon = {
                    IconButton(onClick = onBack) {
                        Icon(Icons.Default.ArrowBack, contentDescription = null)
                    }
                },
                actions = {
                    IconButton(onClick = { showAbout = true }) {
                        Icon(Icons.Default.Info, contentDescription = stringResource(R.string.info))
                    }
                }
            )
        }
    ) { padding ->
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(padding)
                .padding(24.dp),
            verticalArrangement = Arrangement.spacedBy(24.dp)
        ) {
            // Category Reset
            Card(
                modifier = Modifier.fillMaxWidth(),
                onClick = { showKosherDialog = true }
            ) {
                Column(modifier = Modifier.padding(16.dp)) {
                    Text(stringResource(R.string.change_category), style = MaterialTheme.typography.titleMedium)
                    Text(
                        stringResource(R.string.kosher_warning),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.error
                    )
                }
            }

            // Seek Speed
            Column {
                Text(stringResource(R.string.seek_speed), style = MaterialTheme.typography.titleMedium)
                Slider(
                    value = seekSpeed,
                    onValueChange = { seekSpeed = it },
                    onValueChangeFinished = { viewModel.updateSeekSpeed(seekSpeed.toLong()) },
                    valueRange = 1000f..10000f,
                    steps = 9
                )
                Text("${seekSpeed.toInt() / 1000}s per tick", style = MaterialTheme.typography.bodySmall)
            }
        }
    }

    if (showKosherDialog) {
        AlertDialog(
            onDismissRequest = { showKosherDialog = false },
            title = { Text(stringResource(R.string.change_category)) },
            text = { Text(stringResource(R.string.kosher_warning)) },
            confirmButton = {
                TextButton(
                    onClick = {
                        showKosherDialog = false
                        onResetCategory()
                    }
                ) {
                    Text(stringResource(R.string.confirm))
                }
            },
            dismissButton = {
                TextButton(onClick = { showKosherDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }
}
