package com.abloq.purix

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.viewModels
import androidx.compose.runtime.*
import androidx.compose.runtime.livedata.observeAsState
import com.abloq.purix.ui.screens.*
import com.abloq.purix.ui.theme.PuriXTheme
import com.abloq.purix.utils.ManifestLoader
import com.abloq.purix.viewmodel.MediaViewModel

enum class Screen {
    CategorySelection,
    MediaPlayer,
    Settings,
    Queue
}

class MainActivity : ComponentActivity() {

    private val viewModel: MediaViewModel by viewModels()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        
        setContent {
            PuriXTheme {
                val currentCategory by viewModel.currentCategory.observeAsState()
                var currentScreen by remember { 
                    mutableStateOf(if (currentCategory == null) Screen.CategorySelection else Screen.MediaPlayer) 
                }

                // Sync currentScreen with currentCategory for initial load
                LaunchedEffect(currentCategory) {
                    if (currentCategory == null) {
                        currentScreen = Screen.CategorySelection
                    } else if (currentScreen == Screen.CategorySelection) {
                        currentScreen = Screen.MediaPlayer
                    }
                }

                when (currentScreen) {
                    Screen.CategorySelection -> {
                        val categories = remember { ManifestLoader.loadMainManifest(this@MainActivity) }
                        CategorySelectionScreen(
                            categories = categories,
                            onCategorySelected = { category ->
                                viewModel.selectCategory(category)
                                currentScreen = Screen.MediaPlayer
                            }
                        )
                    }
                    Screen.MediaPlayer -> {
                        MediaPlayerScreen(
                            viewModel = viewModel,
                            onOpenSettings = { currentScreen = Screen.Settings },
                            onOpenQueue = { currentScreen = Screen.Queue }
                        )
                    }
                    Screen.Settings -> {
                        SettingsScreen(
                            viewModel = viewModel,
                            onBack = { currentScreen = Screen.MediaPlayer },
                            onResetCategory = { 
                                viewModel.clearCategory()
                                currentScreen = Screen.CategorySelection
                            }
                        )
                    }
                    Screen.Queue -> {
                        QueueScreen(
                            viewModel = viewModel,
                            onDismiss = { currentScreen = Screen.MediaPlayer }
                        )
                    }
                }
            }
        }
    }
}