import React, { useState } from 'react';
import { db } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Search, Info, Wifi, WifiOff, Cpu } from 'lucide-react';

export default function TechnicalInfoView() {
  const inventory = useLiveQuery(() => db.inventory.toArray());
  const [searchTerm, setSearchTerm] = useState('');

  const filteredItems = inventory?.filter(item => 
    item.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    item.technicalInfo.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <header>
        <h2 className="text-3xl font-bold tracking-tight">מידע טכני וגרסאות 📚</h2>
        <p className="text-stone-500">פירוט טכני עבור המוכר להסבר ללקוחות.</p>
      </header>

      <div className="relative">
        <Search className="absolute right-4 top-1/2 -translate-y-1/2 text-stone-400" size={20} />
        <input 
          type="text" 
          placeholder="חפש דגם או מפרט..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          className="w-full bg-white border border-stone-200 rounded-2xl pr-12 pl-4 py-4 outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
        />
      </div>

      <div className="space-y-4">
        {filteredItems?.map((item) => (
          <div key={item.id} className="bg-white p-6 rounded-3xl border border-stone-100 shadow-sm flex flex-col md:flex-row gap-6">
            <div className="md:w-1/3">
              <h3 className="text-xl font-bold text-stone-900 mb-1">{item.name}</h3>
              <p className="text-sm text-blue-600 font-bold mb-4">{item.category}</p>
              
              <div className="flex flex-wrap gap-2">
                <div className="bg-stone-100 px-3 py-1 rounded-lg flex items-center gap-2 text-xs font-bold">
                  <Cpu size={14} /> גרסה: {item.version || 'N/A'}
                </div>
                <div className={`px-3 py-1 rounded-lg flex items-center gap-2 text-xs font-bold ${item.hasWifi ? 'bg-emerald-100 text-emerald-700' : 'bg-stone-100 text-stone-400'}`}>
                  {item.hasWifi ? <Wifi size={14} /> : <WifiOff size={14} />}
                  {item.hasWifi ? 'כולל WiFi' : 'ללא WiFi'}
                </div>
              </div>
            </div>
            
            <div className="flex-1 bg-stone-50 p-6 rounded-2xl relative">
              <Info className="absolute top-4 left-4 text-stone-200" size={24} />
              <h4 className="text-sm font-bold text-stone-400 uppercase tracking-wider mb-2">מפרט טכני מלא:</h4>
              <p className="text-stone-700 whitespace-pre-wrap leading-relaxed">
                {item.technicalInfo || 'לא הוזן מפרט טכני עבור מוצר זה.'}
              </p>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
