import React, { useState } from 'react';
import { db } from '../db';
import { Download, Upload, AlertTriangle, CheckCircle2, FileJson, RefreshCw } from 'lucide-react';

export default function SyncView() {
  const [status, setStatus] = useState<{ type: 'success' | 'error' | 'info'; message: string } | null>(null);
  const [isSyncing, setIsSyncing] = useState(false);

  const exportData = async () => {
    try {
      setIsSyncing(true);
      const data = {
        settings: await db.settings.toArray(),
        orders: await db.orders.toArray(),
        inventory: await db.inventory.toArray(),
        repairs: await db.repairs.toArray(),
        sales: await db.sales.toArray(),
        exportDate: new Date().toISOString()
      };

      const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      link.download = `inventory_backup_${new Date().toISOString().split('T')[0]}.json`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      URL.revokeObjectURL(url);
      
      setStatus({ type: 'success', message: 'הנתונים יוצאו בהצלחה לקובץ!' });
    } catch (err) {
      setStatus({ type: 'error', message: 'שגיאה בייצוא הנתונים.' });
    } finally {
      setIsSyncing(false);
    }
  };

  const importData = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (!confirm('ייבוא נתונים יחליף את הנתונים הקיימים במכשיר זה. האם להמשיך?')) return;

    try {
      setIsSyncing(true);
      const reader = new FileReader();
      reader.onload = async (event) => {
        try {
          const data = JSON.parse(event.target?.result as string);
          
          // Clear and re-populate
          await db.transaction('rw', [db.settings, db.orders, db.inventory, db.repairs, db.sales], async () => {
            await db.settings.clear();
            await db.orders.clear();
            await db.inventory.clear();
            await db.repairs.clear();
            await db.sales.clear();

            if (data.settings) await db.settings.bulkAdd(data.settings);
            if (data.orders) await db.orders.bulkAdd(data.orders);
            if (data.inventory) await db.inventory.bulkAdd(data.inventory);
            if (data.repairs) await db.repairs.bulkAdd(data.repairs);
            if (data.sales) await db.sales.bulkAdd(data.sales);
          });

          setStatus({ type: 'success', message: 'הנתונים יובאו בהצלחה! המערכת מעודכנת.' });
          setTimeout(() => window.location.reload(), 2000);
        } catch (err) {
          setStatus({ type: 'error', message: 'קובץ לא תקין או שגיאה בייבוא.' });
        }
      };
      reader.readAsText(file);
    } catch (err) {
      setStatus({ type: 'error', message: 'שגיאה בקריאת הקובץ.' });
    } finally {
      setIsSyncing(false);
    }
  };

  return (
    <div className="space-y-8">
      <header>
        <h2 className="text-3xl font-bold tracking-tight">סנכרון וגיבוי אופליין 🔄</h2>
        <p className="text-stone-500">העבר נתונים בין מכשירים ללא צורך באינטרנט.</p>
      </header>

      {status && (
        <div className={`p-6 rounded-2xl flex items-center gap-4 animate-in slide-in-from-right-4 ${
          status.type === 'success' ? 'bg-emerald-50 text-emerald-700 border border-emerald-100' : 
          status.type === 'error' ? 'bg-red-50 text-red-700 border border-red-100' : 
          'bg-blue-50 text-blue-700 border border-blue-100'
        }`}>
          {status.type === 'success' ? <CheckCircle2 size={24} /> : <AlertTriangle size={24} />}
          <p className="font-bold">{status.message}</p>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
        <div className="bg-white p-10 rounded-3xl shadow-sm border border-stone-100 flex flex-col items-center text-center space-y-6">
          <div className="w-20 h-20 bg-emerald-100 text-emerald-600 rounded-full flex items-center justify-center">
            <Download size={40} />
          </div>
          <div>
            <h3 className="text-2xl font-bold mb-2">ייצוא נתונים</h3>
            <p className="text-stone-500">צור קובץ גיבוי עם כל המידע הקיים במכשיר זה.</p>
          </div>
          <button 
            disabled={isSyncing}
            onClick={exportData}
            className="w-full bg-emerald-600 text-white py-4 rounded-2xl font-bold shadow-lg hover:bg-emerald-700 transition-all flex items-center justify-center gap-2"
          >
            {isSyncing ? <RefreshCw className="animate-spin" /> : <FileJson />}
            ייצא קובץ סנכרון
          </button>
        </div>

        <div className="bg-white p-10 rounded-3xl shadow-sm border border-stone-100 flex flex-col items-center text-center space-y-6">
          <div className="w-20 h-20 bg-blue-100 text-blue-600 rounded-full flex items-center justify-center">
            <Upload size={40} />
          </div>
          <div>
            <h3 className="text-2xl font-bold mb-2">ייבוא נתונים</h3>
            <p className="text-stone-500">טען קובץ גיבוי ממכשיר אחר לעדכון המערכת.</p>
          </div>
          <label className="w-full bg-blue-600 text-white py-4 rounded-2xl font-bold shadow-lg hover:bg-blue-700 transition-all flex items-center justify-center gap-2 cursor-pointer">
            <Upload />
            בחר קובץ לייבוא
            <input 
              type="file" 
              accept=".json" 
              className="hidden" 
              onChange={importData}
              disabled={isSyncing}
            />
          </label>
        </div>
      </div>

      <div className="bg-amber-50 p-8 rounded-3xl border border-amber-100 flex gap-6 items-start">
        <div className="bg-amber-100 p-3 rounded-2xl text-amber-600">
          <AlertTriangle size={24} />
        </div>
        <div className="space-y-2">
          <h4 className="font-bold text-amber-900">שים לב!</h4>
          <p className="text-sm text-amber-800 leading-relaxed">
            מכיוון שהמכשירים חסומים לאינטרנט, הסנכרון מתבצע ידנית. מומלץ לבצע ייצוא וייבוא פעם ביום כדי לשמור על שני המכשירים מעודכנים. 
            פעולת הייבוא דורסת את המידע הקיים במכשיר זה ומחליפה אותו במידע מהקובץ.
          </p>
        </div>
      </div>
    </div>
  );
}
