import React, { useState } from 'react';
import { db } from '../db';
import { ChevronLeft, ChevronRight, Check, Plus, Trash2 } from 'lucide-react';
import { motion, AnimatePresence } from 'motion/react';

interface SetupWizardProps {
  onComplete: () => void;
}

export default function SetupWizard({ onComplete }: SetupWizardProps) {
  const [step, setStep] = useState(1);
  const [categories, setCategories] = useState<string[]>(['סמארטפונים', 'טאבלטים', 'אביזרים']);
  const [stages, setStages] = useState<{ name: string; alert: string }[]>([
    { name: 'התקבל לטיפול', alert: 'מיידי' },
    { name: 'בבדיקה', alert: 'ללא' },
    { name: 'בצריבה/תיקון', alert: 'ללא' },
    { name: 'מוכן למסירה', alert: 'מיידי' }
  ]);
  const [newCat, setNewCat] = useState('');
  const [newStage, setNewStage] = useState({ name: '', alert: 'ללא' });

  const handleComplete = async () => {
    await db.settings.add({ key: 'categories', value: categories });
    await db.settings.add({ key: 'work_stages', value: stages });
    await db.settings.add({ key: 'wizard_complete', value: true });
    onComplete();
  };

  return (
    <div className="min-h-screen bg-emerald-700 flex items-center justify-center p-4">
      <motion.div 
        initial={{ scale: 0.9, opacity: 0 }}
        animate={{ scale: 1, opacity: 1 }}
        className="bg-white w-full max-w-2xl rounded-3xl shadow-2xl overflow-hidden flex flex-col"
      >
        {/* Progress Bar */}
        <div className="h-2 bg-stone-100 flex">
          <div 
            className="h-full bg-emerald-500 transition-all duration-500" 
            style={{ width: `${(step / 3) * 100}%` }}
          />
        </div>

        <div className="p-8 md:p-12 flex-1">
          <AnimatePresence mode="wait">
            {step === 1 && (
              <motion.div
                key="step1"
                initial={{ x: 20, opacity: 0 }}
                animate={{ x: 0, opacity: 1 }}
                exit={{ x: -20, opacity: 0 }}
                className="space-y-6"
              >
                <header>
                  <h2 className="text-3xl font-bold text-stone-900">ברוכים הבאים! 🌟</h2>
                  <p className="text-stone-500 mt-2 text-lg">בוא נגדיר את קטגוריות המוצרים בעסק שלך.</p>
                </header>

                <div className="space-y-4">
                  <div className="flex gap-2">
                    <input 
                      type="text" 
                      value={newCat}
                      onChange={(e) => setNewCat(e.target.value)}
                      placeholder="הוסף קטגוריה חדשה..."
                      className="flex-1 border border-stone-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-emerald-500 outline-none"
                    />
                    <button 
                      onClick={() => {
                        if (newCat) {
                          setCategories([...categories, newCat]);
                          setNewCat('');
                        }
                      }}
                      className="bg-emerald-600 text-white p-3 rounded-xl hover:bg-emerald-700 transition-colors"
                    >
                      <Plus size={24} />
                    </button>
                  </div>

                  <div className="flex flex-wrap gap-2">
                    {categories.map((cat, i) => (
                      <div key={i} className="bg-stone-100 px-4 py-2 rounded-full flex items-center gap-2 group">
                        <span>{cat}</span>
                        <button 
                          onClick={() => setCategories(categories.filter((_, idx) => idx !== i))}
                          className="text-stone-400 hover:text-red-500 transition-colors"
                        >
                          <Trash2 size={14} />
                        </button>
                      </div>
                    ))}
                  </div>
                </div>
              </motion.div>
            )}

            {step === 2 && (
              <motion.div
                key="step2"
                initial={{ x: 20, opacity: 0 }}
                animate={{ x: 0, opacity: 1 }}
                exit={{ x: -20, opacity: 0 }}
                className="space-y-6"
              >
                <header>
                  <h2 className="text-3xl font-bold text-stone-900">שלבי עבודה 🛠️</h2>
                  <p className="text-stone-500 mt-2 text-lg">מהם השלבים שמוצר עובר מרגע הקליטה ועד המסירה?</p>
                </header>

                <div className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                    <input 
                      type="text" 
                      value={newStage.name}
                      onChange={(e) => setNewStage({ ...newStage, name: e.target.value })}
                      placeholder="שם השלב (למשל: בתיקון)"
                      className="border border-stone-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-emerald-500 outline-none"
                    />
                    <div className="flex gap-2">
                      <select 
                        value={newStage.alert}
                        onChange={(e) => setNewStage({ ...newStage, alert: e.target.value })}
                        className="flex-1 border border-stone-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-emerald-500 outline-none"
                      >
                        <option value="ללא">ללא התראה</option>
                        <option value="מיידי">התראה מיידית</option>
                        <option value="יום">אחרי יום</option>
                      </select>
                      <button 
                        onClick={() => {
                          if (newStage.name) {
                            setStages([...stages, newStage]);
                            setNewStage({ name: '', alert: 'ללא' });
                          }
                        }}
                        className="bg-emerald-600 text-white p-3 rounded-xl hover:bg-emerald-700 transition-colors"
                      >
                        <Plus size={24} />
                      </button>
                    </div>
                  </div>

                  <div className="space-y-2">
                    {stages.map((stage, i) => (
                      <div key={i} className="bg-stone-50 border border-stone-100 p-4 rounded-xl flex justify-between items-center">
                        <div>
                          <span className="font-bold text-stone-700">{i + 1}. {stage.name}</span>
                          <span className="text-xs text-stone-400 mr-2">({stage.alert})</span>
                        </div>
                        <button 
                          onClick={() => setStages(stages.filter((_, idx) => idx !== i))}
                          className="text-stone-300 hover:text-red-500 transition-colors"
                        >
                          <Trash2 size={18} />
                        </button>
                      </div>
                    ))}
                  </div>
                </div>
              </motion.div>
            )}

            {step === 3 && (
              <motion.div
                key="step3"
                initial={{ x: 20, opacity: 0 }}
                animate={{ x: 0, opacity: 1 }}
                exit={{ x: -20, opacity: 0 }}
                className="space-y-8 text-center"
              >
                <div className="w-24 h-24 bg-emerald-100 text-emerald-600 rounded-full flex items-center justify-center mx-auto mb-6">
                  <Check size={48} />
                </div>
                <header>
                  <h2 className="text-3xl font-bold text-stone-900">הכל מוכן! ✨</h2>
                  <p className="text-stone-500 mt-2 text-lg">הגדרנו את הבסיס. עכשיו אפשר להתחיל לנהל את העסק בצורה חכמה.</p>
                </header>
                <div className="bg-stone-50 p-6 rounded-2xl text-right">
                  <h4 className="font-bold mb-2">סיכום הגדרות:</h4>
                  <p className="text-sm text-stone-600">• {categories.length} קטגוריות מוצרים</p>
                  <p className="text-sm text-stone-600">• {stages.length} שלבי עבודה מוגדרים</p>
                </div>
              </motion.div>
            )}
          </AnimatePresence>
        </div>

        <div className="p-8 bg-stone-50 flex justify-between items-center">
          {step > 1 ? (
            <button 
              onClick={() => setStep(step - 1)}
              className="flex items-center gap-2 text-stone-500 font-bold hover:text-stone-800 transition-colors"
            >
              <ChevronRight size={20} /> חזרה
            </button>
          ) : <div></div>}

          {step < 3 ? (
            <button 
              onClick={() => setStep(step + 1)}
              className="bg-emerald-600 text-white px-8 py-3 rounded-xl font-bold shadow-lg hover:bg-emerald-700 transition-all flex items-center gap-2"
            >
              המשך <ChevronLeft size={20} />
            </button>
          ) : (
            <button 
              onClick={handleComplete}
              className="bg-emerald-600 text-white px-8 py-3 rounded-xl font-bold shadow-lg hover:bg-emerald-700 transition-all flex items-center gap-2"
            >
              יאללה, בוא נתחיל! <Check size={20} />
            </button>
          )}
        </div>
      </motion.div>
    </div>
  );
}
