import React, { useState } from 'react';
import { db, type Repair } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Plus, Store, User, MapPin, Clock, Package, CheckCircle2 } from 'lucide-react';

export default function SellerView() {
  const repairs = useLiveQuery(() => db.repairs.toArray());
  const inventory = useLiveQuery(() => db.inventory.toArray());
  const stages = useLiveQuery(() => db.settings.where('key').equals('work_stages').first());
  
  const [isAdding, setIsAdding] = useState(false);
  const [newRepair, setNewRepair] = useState<Partial<Repair>>({
    customerName: '',
    productName: '',
    location: '',
    status: '',
    progress: 0
  });

  // Get unique locations for suggestions from all sources
  const existingLocations = Array.from(new Set([
    ...(repairs?.map(r => r.location) || []),
    ...(inventory?.map(i => i.location) || [])
  ])).filter(Boolean);

  const handleAddRepair = async (e: React.FormEvent) => {
    e.preventDefault();
    if (newRepair.customerName && newRepair.productName) {
      await db.repairs.add({
        customerName: newRepair.customerName!,
        productName: newRepair.productName!,
        location: newRepair.location || '',
        intakeDate: new Date(),
        status: stages?.value[0]?.name || 'התקבל',
        progress: 0
      });
      setIsAdding(false);
      setNewRepair({ customerName: '', productName: '', location: '', status: '', progress: 0 });
    }
  };

  return (
    <div className="space-y-8">
      <header className="flex justify-between items-center">
        <div>
          <h2 className="text-3xl font-bold tracking-tight">חלון מוכר (מכירות ושירות) 🏪</h2>
          <p className="text-stone-500">קליטת מוצרים חדשים ומעקב אחר סטטוס מוכן.</p>
        </div>
        <button 
          onClick={() => setIsAdding(true)}
          className="bg-blue-600 text-white px-6 py-3 rounded-xl font-bold shadow-md hover:bg-blue-700 transition-colors flex items-center gap-2"
        >
          <Plus size={20} /> קליטת מוצר לטיפול
        </button>
      </header>

      {isAdding && (
        <div className="bg-white p-8 rounded-3xl shadow-xl border border-blue-100 animate-in slide-in-from-top-4">
          <h3 className="text-xl font-bold mb-6">קליטת מוצר חדש מלקוח</h3>
          <form onSubmit={handleAddRepair} className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">שם הלקוח</label>
              <div className="relative">
                <User className="absolute right-3 top-1/2 -translate-y-1/2 text-stone-400" size={18} />
                <input 
                  required
                  type="text" 
                  value={newRepair.customerName}
                  onChange={(e) => setNewRepair({ ...newRepair, customerName: e.target.value })}
                  className="w-full border border-stone-200 rounded-xl pr-10 pl-4 py-3 outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">שם המכשיר/המוצר</label>
              <div className="relative">
                <Package className="absolute right-3 top-1/2 -translate-y-1/2 text-stone-400" size={18} />
                <input 
                  required
                  type="text" 
                  value={newRepair.productName}
                  onChange={(e) => setNewRepair({ ...newRepair, productName: e.target.value })}
                  className="w-full border border-stone-200 rounded-xl pr-10 pl-4 py-3 outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
            </div>
            <div className="space-y-1 md:col-span-2">
              <label className="text-sm font-bold text-stone-700">איפה הנחת אותו? (מיקום בבית)</label>
              <div className="relative">
                <MapPin className="absolute right-3 top-1/2 -translate-y-1/2 text-stone-400" size={18} />
                <input 
                  type="text" 
                  list="locations"
                  value={newRepair.location}
                  onChange={(e) => setNewRepair({ ...newRepair, location: e.target.value })}
                  placeholder="למשל: מגירה עליונה, שקית אדומה"
                  className="w-full border border-stone-200 rounded-xl pr-10 pl-4 py-3 outline-none focus:ring-2 focus:ring-blue-500"
                />
                <datalist id="locations">
                  {existingLocations.map(loc => <option key={loc} value={loc} />)}
                </datalist>
              </div>
            </div>
            <div className="md:col-span-2 flex justify-end gap-3">
              <button 
                type="button"
                onClick={() => setIsAdding(false)}
                className="px-8 py-3 text-stone-500 font-bold"
              >
                ביטול
              </button>
              <button 
                type="submit"
                className="bg-blue-600 text-white px-10 py-3 rounded-xl font-bold shadow-lg hover:bg-blue-700 transition-all"
              >
                קלוט מוצר
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Repairs Tracking */}
        <div className="space-y-4">
          <h3 className="text-xl font-bold flex items-center gap-2">
            <Clock className="text-blue-600" /> מוצרים בטיפול
          </h3>
          <div className="space-y-3">
            {repairs?.filter(r => r.progress < 100).map((repair) => (
              <div key={repair.id} className="bg-white p-5 rounded-2xl border border-stone-100 shadow-sm">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h4 className="font-bold text-stone-800">{repair.productName}</h4>
                    <p className="text-xs text-stone-500">לקוח: {repair.customerName}</p>
                  </div>
                  <span className="bg-blue-50 text-blue-700 text-[10px] font-bold px-2 py-1 rounded-full uppercase">
                    {repair.status}
                  </span>
                </div>
                <div className="space-y-1">
                  <div className="flex justify-between text-[10px] font-bold text-stone-400">
                    <span>התקדמות</span>
                    <span>{repair.progress}%</span>
                  </div>
                  <div className="h-2 bg-stone-100 rounded-full overflow-hidden">
                    <div 
                      className="h-full bg-blue-500 transition-all duration-500"
                      style={{ width: `${repair.progress}%` }}
                    />
                  </div>
                </div>
              </div>
            ))}
            {repairs?.filter(r => r.progress < 100).length === 0 && (
              <p className="text-stone-400 text-sm italic">אין מוצרים בטיפול כרגע.</p>
            )}
          </div>
        </div>

        {/* Ready for Pickup */}
        <div className="space-y-4">
          <h3 className="text-xl font-bold flex items-center gap-2">
            <CheckCircle2 className="text-emerald-600" /> מוכן למסירה
          </h3>
          <div className="space-y-3">
            {repairs?.filter(r => r.progress === 100).map((repair) => (
              <div key={repair.id} className="bg-emerald-50 p-5 rounded-2xl border border-emerald-100 shadow-sm flex justify-between items-center">
                <div>
                  <h4 className="font-bold text-emerald-900">{repair.productName}</h4>
                  <p className="text-xs text-emerald-700">לקוח: {repair.customerName}</p>
                  <div className="flex items-center gap-1 text-[10px] text-emerald-600 mt-1">
                    <MapPin size={10} /> נמצא ב: {repair.location}
                  </div>
                </div>
                <button 
                  onClick={async () => {
                    if (confirm('האם המוצר נמסר ללקוח?')) {
                      await db.repairs.delete(repair.id!);
                    }
                  }}
                  className="bg-emerald-600 text-white px-4 py-2 rounded-xl text-xs font-bold shadow-sm"
                >
                  נמסר
                </button>
              </div>
            ))}
            {repairs?.filter(r => r.progress === 100).length === 0 && (
              <p className="text-stone-400 text-sm italic">אין מוצרים מוכנים למסירה.</p>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
