import React, { useState } from 'react';
import { db, type Order } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Plus, Package, Clock, CheckCircle, Trash2, AlertCircle } from 'lucide-react';
import { format, isPast, isToday } from 'date-fns';
import { he } from 'date-fns/locale';

export default function OrdersView() {
  const orders = useLiveQuery(() => db.orders.toArray());
  const [isAdding, setIsAdding] = useState(false);
  const [newOrder, setNewOrder] = useState<Partial<Order>>({
    productName: '',
    estimatedArrival: new Date(),
    collectionCode: '',
    status: 'pending'
  });

  const handleAddOrder = async (e: React.FormEvent) => {
    e.preventDefault();
    if (newOrder.productName) {
      await db.orders.add({
        productName: newOrder.productName!,
        orderDate: new Date(),
        estimatedArrival: new Date(newOrder.estimatedArrival!),
        collectionCode: newOrder.collectionCode || '',
        status: 'pending'
      });
      setIsAdding(false);
      setNewOrder({ productName: '', estimatedArrival: new Date(), collectionCode: '', status: 'pending' });
    }
  };

  const updateStatus = async (id: number, status: Order['status']) => {
    await db.orders.update(id, { status });
  };

  const deleteOrder = async (id: number) => {
    if (confirm('האם למחוק את ההזמנה?')) {
      await db.orders.delete(id);
    }
  };

  return (
    <div className="space-y-6">
      <header className="flex justify-between items-center">
        <div>
          <h2 className="text-3xl font-bold tracking-tight">הזמנות מחו"ל 📦</h2>
          <p className="text-stone-500">עקוב אחר רכיבים ומוצרים שהוזמנו.</p>
        </div>
        <button 
          onClick={() => setIsAdding(true)}
          className="bg-emerald-600 text-white px-6 py-3 rounded-xl font-bold shadow-md hover:bg-emerald-700 transition-colors flex items-center gap-2"
        >
          <Plus size={20} /> הזמנה חדשה
        </button>
      </header>

      {isAdding && (
        <div className="bg-white p-6 rounded-2xl shadow-md border border-emerald-100 animate-in fade-in slide-in-from-top-4">
          <form onSubmit={handleAddOrder} className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">שם המוצר</label>
              <input 
                required
                type="text" 
                value={newOrder.productName}
                onChange={(e) => setNewOrder({ ...newOrder, productName: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-2 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">תאריך הגעה משוער</label>
              <input 
                required
                type="date" 
                onChange={(e) => setNewOrder({ ...newOrder, estimatedArrival: new Date(e.target.value) })}
                className="w-full border border-stone-200 rounded-xl px-4 py-2 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">קוד איסוף</label>
              <input 
                type="text" 
                value={newOrder.collectionCode}
                onChange={(e) => setNewOrder({ ...newOrder, collectionCode: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-2 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="md:col-span-3 flex justify-end gap-2 mt-2">
              <button 
                type="button"
                onClick={() => setIsAdding(false)}
                className="px-6 py-2 text-stone-500 font-bold"
              >
                ביטול
              </button>
              <button 
                type="submit"
                className="bg-emerald-600 text-white px-8 py-2 rounded-xl font-bold shadow-md"
              >
                שמור הזמנה
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="grid grid-cols-1 gap-4">
        {orders?.length === 0 && !isAdding && (
          <div className="text-center py-20 bg-white rounded-3xl border border-dashed border-stone-200">
            <Package size={48} className="mx-auto text-stone-300 mb-4" />
            <p className="text-stone-500">אין הזמנות פעילות כרגע.</p>
          </div>
        )}

        {orders?.sort((a, b) => a.estimatedArrival.getTime() - b.estimatedArrival.getTime()).map((order) => {
          const isDue = isToday(order.estimatedArrival) || (isPast(order.estimatedArrival) && order.status === 'pending');
          
          return (
            <div 
              key={order.id} 
              className={`bg-white p-6 rounded-2xl shadow-sm border flex flex-col md:flex-row justify-between items-center gap-4 transition-all ${
                isDue ? 'border-amber-200 bg-amber-50/30' : 'border-stone-100'
              }`}
            >
              <div className="flex items-center gap-4 flex-1">
                <div className={`p-3 rounded-xl ${isDue ? 'bg-amber-100 text-amber-600' : 'bg-stone-100 text-stone-500'}`}>
                  <Package size={24} />
                </div>
                <div>
                  <h4 className="font-bold text-lg">{order.productName}</h4>
                  <div className="flex items-center gap-3 text-sm text-stone-500 mt-1">
                    <span className="flex items-center gap-1">
                      <Clock size={14} /> הגעה: {format(order.estimatedArrival, 'dd/MM/yyyy', { locale: he })}
                    </span>
                    {order.collectionCode && (
                      <span className="bg-stone-100 px-2 py-0.5 rounded text-xs font-mono">
                        קוד: {order.collectionCode}
                      </span>
                    )}
                  </div>
                </div>
              </div>

              {isDue && order.status === 'pending' && (
                <div className="flex items-center gap-2 text-amber-600 bg-amber-100 px-4 py-2 rounded-xl text-sm font-bold animate-pulse">
                  <AlertCircle size={18} /> הגיע זמן איסוף!
                </div>
              )}

              <div className="flex items-center gap-2">
                {order.status === 'pending' ? (
                  <button 
                    onClick={() => updateStatus(order.id!, 'arrived')}
                    className="bg-emerald-50 text-emerald-700 px-4 py-2 rounded-xl text-sm font-bold hover:bg-emerald-100 transition-colors"
                  >
                    סמן כהגיע
                  </button>
                ) : order.status === 'arrived' ? (
                  <button 
                    onClick={() => updateStatus(order.id!, 'collected')}
                    className="bg-blue-50 text-blue-700 px-4 py-2 rounded-xl text-sm font-bold hover:bg-blue-100 transition-colors"
                  >
                    סמן כנאסף
                  </button>
                ) : (
                  <span className="text-emerald-600 flex items-center gap-1 text-sm font-bold">
                    <CheckCircle size={18} /> נאסף
                  </span>
                )}
                
                <button 
                  onClick={() => deleteOrder(order.id!)}
                  className="p-2 text-stone-300 hover:text-red-500 transition-colors"
                >
                  <Trash2 size={18} />
                </button>
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
}
