import React, { useState } from 'react';
import { db, type Repair } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Wrench, CheckCircle2, Clock, MapPin, ChevronRight, Save } from 'lucide-react';

export default function ManagerView() {
  const repairs = useLiveQuery(() => db.repairs.toArray());
  const stages = useLiveQuery(() => db.settings.where('key').equals('work_stages').first());
  const [editingId, setEditingId] = useState<number | null>(null);

  const updateRepair = async (id: number, updates: Partial<Repair>) => {
    await db.repairs.update(id, updates);
  };

  return (
    <div className="space-y-6">
      <header>
        <h2 className="text-3xl font-bold tracking-tight">חלון מנהל (מעבדה) 👨‍🔧</h2>
        <p className="text-stone-500">עדכן התקדמות עבודה ושלבי טיפול.</p>
      </header>

      <div className="grid grid-cols-1 gap-4">
        {repairs?.filter(r => r.progress < 100).map((repair) => (
          <div key={repair.id} className="bg-white p-6 rounded-3xl shadow-sm border border-stone-100 flex flex-col gap-6">
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
              <div className="flex items-center gap-4">
                <div className="bg-blue-100 text-blue-600 p-3 rounded-2xl">
                  <Wrench size={24} />
                </div>
                <div>
                  <h4 className="text-xl font-bold">{repair.productName}</h4>
                  <p className="text-stone-500 text-sm">לקוח: {repair.customerName} | מיקום: {repair.location}</p>
                </div>
              </div>
              
              <div className="flex items-center gap-3">
                <select 
                  value={repair.status}
                  onChange={(e) => updateRepair(repair.id!, { status: e.target.value })}
                  className="bg-stone-100 border-none rounded-xl px-4 py-2 text-sm font-bold outline-none focus:ring-2 focus:ring-blue-500"
                >
                  {stages?.value.map((s: any) => (
                    <option key={s.name} value={s.name}>{s.name}</option>
                  ))}
                </select>
                <button 
                  onClick={() => updateRepair(repair.id!, { progress: 100, status: 'מוכן למסירה' })}
                  className="bg-emerald-600 text-white p-2 rounded-xl hover:bg-emerald-700 transition-colors"
                  title="סמן כהושלם"
                >
                  <CheckCircle2 size={20} />
                </button>
              </div>
            </div>

            <div className="space-y-2">
              <div className="flex justify-between text-sm font-bold">
                <span className="text-stone-500">התקדמות עבודה</span>
                <span className="text-blue-600">{repair.progress}%</span>
              </div>
              <div className="h-4 bg-stone-100 rounded-full overflow-hidden">
                <div 
                  className="h-full bg-blue-500 transition-all duration-500"
                  style={{ width: `${repair.progress}%` }}
                />
              </div>
              <input 
                type="range" 
                min="0" 
                max="100" 
                value={repair.progress}
                onChange={(e) => updateRepair(repair.id!, { progress: parseInt(e.target.value) })}
                className="w-full h-2 bg-stone-100 rounded-lg appearance-none cursor-pointer accent-blue-600"
              />
            </div>
          </div>
        ))}

        {repairs?.filter(r => r.progress < 100).length === 0 && (
          <div className="text-center py-20 bg-white rounded-3xl border border-dashed border-stone-200">
            <CheckCircle2 size={48} className="mx-auto text-emerald-200 mb-4" />
            <p className="text-stone-500 font-bold">אין עבודות פעילות בתיקון. הכל מוכן!</p>
          </div>
        )}
      </div>
    </div>
  );
}
