import React, { useState } from 'react';
import { db, type InventoryItem } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Plus, Search, MapPin, Info, Wifi, WifiOff, Trash2, Edit3, Package } from 'lucide-react';

export default function InventoryView() {
  const inventory = useLiveQuery(() => db.inventory.toArray());
  const repairs = useLiveQuery(() => db.repairs.toArray());
  const categories = useLiveQuery(() => db.settings.where('key').equals('categories').first());
  const [isAdding, setIsAdding] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [newItem, setNewItem] = useState<Partial<InventoryItem>>({
    name: '',
    category: '',
    technicalInfo: '',
    version: '',
    hasWifi: false,
    quantity: 1,
    location: ''
  });

  // Get unique locations for suggestions from all sources
  const existingLocations = Array.from(new Set([
    ...(repairs?.map(r => r.location) || []),
    ...(inventory?.map(i => i.location) || [])
  ])).filter(Boolean);

  const handleAddItem = async (e: React.FormEvent) => {
    e.preventDefault();
    if (newItem.name && newItem.category) {
      await db.inventory.add({
        name: newItem.name!,
        category: newItem.category!,
        technicalInfo: newItem.technicalInfo || '',
        version: newItem.version || '',
        hasWifi: newItem.hasWifi || false,
        quantity: newItem.quantity || 1,
        location: newItem.location || ''
      });
      setIsAdding(false);
      setNewItem({ name: '', category: '', technicalInfo: '', version: '', hasWifi: false, quantity: 1, location: '' });
    }
  };

  const deleteItem = async (id: number) => {
    if (confirm('האם למחוק פריט זה מהמלאי?')) {
      await db.inventory.delete(id);
    }
  };

  const filteredInventory = inventory?.filter(item => 
    item.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    item.category.toLowerCase().includes(searchTerm.toLowerCase()) ||
    item.location.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <header className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h2 className="text-3xl font-bold tracking-tight">ניהול מלאי 📦</h2>
          <p className="text-stone-500">נהל את מלאי המוצרים והרכיבים שלך.</p>
        </div>
        <button 
          onClick={() => setIsAdding(true)}
          className="bg-emerald-600 text-white px-6 py-3 rounded-xl font-bold shadow-md hover:bg-emerald-700 transition-colors flex items-center gap-2"
        >
          <Plus size={20} /> הוסף מוצר למלאי
        </button>
      </header>

      <div className="relative">
        <Search className="absolute right-4 top-1/2 -translate-y-1/2 text-stone-400" size={20} />
        <input 
          type="text" 
          placeholder="חפש לפי שם, קטגוריה או מיקום..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          className="w-full bg-white border border-stone-200 rounded-2xl pr-12 pl-4 py-4 outline-none focus:ring-2 focus:ring-emerald-500 shadow-sm"
        />
      </div>

      {isAdding && (
        <div className="bg-white p-8 rounded-3xl shadow-xl border border-emerald-100 animate-in zoom-in-95">
          <h3 className="text-xl font-bold mb-6">הוספת מוצר חדש</h3>
          <form onSubmit={handleAddItem} className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">שם המוצר</label>
              <input 
                required
                type="text" 
                value={newItem.name}
                onChange={(e) => setNewItem({ ...newItem, name: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">קטגוריה</label>
              <select 
                required
                value={newItem.category}
                onChange={(e) => setNewItem({ ...newItem, category: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              >
                <option value="">בחר קטגוריה...</option>
                {categories?.value.map((cat: string) => (
                  <option key={cat} value={cat}>{cat}</option>
                ))}
              </select>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">מיקום בבית (שדה חופשי)</label>
              <input 
                type="text" 
                list="inventory-locations"
                value={newItem.location}
                onChange={(e) => setNewItem({ ...newItem, location: e.target.value })}
                placeholder="למשל: מגירה 2, שקית כחולה"
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              />
              <datalist id="inventory-locations">
                {existingLocations.map(loc => <option key={loc} value={loc} />)}
              </datalist>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">כמות במלאי</label>
              <input 
                type="number" 
                value={newItem.quantity}
                onChange={(e) => setNewItem({ ...newItem, quantity: parseInt(e.target.value) })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">גרסה / מפרט</label>
              <input 
                type="text" 
                value={newItem.version}
                onChange={(e) => setNewItem({ ...newItem, version: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="flex items-center gap-3 pt-8">
              <input 
                type="checkbox" 
                id="wifi"
                checked={newItem.hasWifi}
                onChange={(e) => setNewItem({ ...newItem, hasWifi: e.target.checked })}
                className="w-5 h-5 accent-emerald-600"
              />
              <label htmlFor="wifi" className="font-bold text-stone-700">כולל WiFi?</label>
            </div>
            <div className="md:col-span-2 space-y-1">
              <label className="text-sm font-bold text-stone-700">מידע טכני נוסף</label>
              <textarea 
                value={newItem.technicalInfo}
                onChange={(e) => setNewItem({ ...newItem, technicalInfo: e.target.value })}
                rows={3}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-emerald-500"
              />
            </div>
            <div className="md:col-span-2 flex justify-end gap-3">
              <button 
                type="button"
                onClick={() => setIsAdding(false)}
                className="px-8 py-3 text-stone-500 font-bold"
              >
                ביטול
              </button>
              <button 
                type="submit"
                className="bg-emerald-600 text-white px-10 py-3 rounded-xl font-bold shadow-lg hover:bg-emerald-700 transition-all"
              >
                שמור במלאי
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {filteredInventory?.map((item) => (
          <div key={item.id} className="bg-white rounded-3xl shadow-sm border border-stone-100 overflow-hidden hover:shadow-md transition-shadow">
            <div className="p-6">
              <div className="flex justify-between items-start mb-4">
                <span className="bg-emerald-50 text-emerald-700 text-xs font-bold px-3 py-1 rounded-full">
                  {item.category}
                </span>
                <div className="flex gap-1">
                  <button className="p-2 text-stone-300 hover:text-emerald-600 transition-colors">
                    <Edit3 size={16} />
                  </button>
                  <button 
                    onClick={() => deleteItem(item.id!)}
                    className="p-2 text-stone-300 hover:text-red-500 transition-colors"
                  >
                    <Trash2 size={16} />
                  </button>
                </div>
              </div>
              
              <h4 className="text-xl font-bold mb-2">{item.name}</h4>
              
              <div className="space-y-3 mt-4">
                <div className="flex items-center gap-2 text-stone-600 text-sm">
                  <MapPin size={16} className="text-stone-400" />
                  <span className="font-medium">מיקום: {item.location || 'לא הוגדר'}</span>
                </div>
                <div className="flex items-center gap-2 text-stone-600 text-sm">
                  <Package size={16} className="text-stone-400" />
                  <span className="font-medium">כמות: {item.quantity} יחידות</span>
                </div>
                <div className="flex items-center gap-2 text-stone-600 text-sm">
                  <Info size={16} className="text-stone-400" />
                  <span className="font-medium">גרסה: {item.version || 'N/A'}</span>
                </div>
                <div className="flex items-center gap-2 text-stone-600 text-sm">
                  {item.hasWifi ? (
                    <><Wifi size={16} className="text-emerald-500" /> <span className="text-emerald-600 font-bold">כולל WiFi</span></>
                  ) : (
                    <><WifiOff size={16} className="text-stone-300" /> <span className="text-stone-400">ללא WiFi</span></>
                  )}
                </div>
              </div>
            </div>
            <div className="bg-stone-50 px-6 py-4 border-t border-stone-100">
              <p className="text-xs text-stone-400 line-clamp-2 italic">
                {item.technicalInfo || 'אין מידע טכני נוסף'}
              </p>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
