import React, { useState } from 'react';
import { db, type Sale } from '../db';
import { useLiveQuery } from 'dexie-react-hooks';
import { Database, Plus, DollarSign, MapPin, Calendar, Trash2 } from 'lucide-react';
import { format } from 'date-fns';
import { he } from 'date-fns/locale';

export default function CashierView() {
  const sales = useLiveQuery(() => db.sales.toArray());
  const [isAdding, setIsAdding] = useState(false);
  const [newSale, setNewSale] = useState<Partial<Sale>>({
    productName: '',
    price: 0,
    moneyLocation: '',
    notes: ''
  });

  const handleAddSale = async (e: React.FormEvent) => {
    e.preventDefault();
    if (newSale.productName && newSale.price) {
      await db.sales.add({
        productName: newSale.productName!,
        price: Number(newSale.price),
        moneyLocation: newSale.moneyLocation || '',
        saleDate: new Date(),
        notes: newSale.notes || ''
      });
      setIsAdding(false);
      setNewSale({ productName: '', price: 0, moneyLocation: '', notes: '' });
    }
  };

  const totalSales = sales?.reduce((sum, sale) => sum + sale.price, 0) || 0;

  return (
    <div className="space-y-6">
      <header className="flex justify-between items-center">
        <div>
          <h2 className="text-3xl font-bold tracking-tight">קופה ודיווח מכירות 💰</h2>
          <p className="text-stone-500">תיעוד מכירות וניהול הכסף הפיזי.</p>
        </div>
        <button 
          onClick={() => setIsAdding(true)}
          className="bg-amber-600 text-white px-6 py-3 rounded-xl font-bold shadow-md hover:bg-amber-700 transition-colors flex items-center gap-2"
        >
          <Plus size={20} /> רישום מכירה
        </button>
      </header>

      <div className="bg-white p-8 rounded-3xl shadow-sm border border-stone-100 flex items-center justify-between">
        <div>
          <p className="text-sm font-bold text-stone-400 uppercase tracking-wider">סה"כ מכירות (כל הזמן)</p>
          <h3 className="text-5xl font-black text-amber-600 mt-2">₪{totalSales.toLocaleString()}</h3>
        </div>
        <div className="bg-amber-50 p-4 rounded-2xl text-amber-600">
          <Database size={48} />
        </div>
      </div>

      {isAdding && (
        <div className="bg-white p-8 rounded-3xl shadow-xl border border-amber-100 animate-in zoom-in-95">
          <h3 className="text-xl font-bold mb-6">רישום מכירה חדשה</h3>
          <form onSubmit={handleAddSale} className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">שם המוצר שנמכר</label>
              <input 
                required
                type="text" 
                value={newSale.productName}
                onChange={(e) => setNewSale({ ...newSale, productName: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-amber-500"
              />
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">מחיר (₪)</label>
              <div className="relative">
                <DollarSign className="absolute right-3 top-1/2 -translate-y-1/2 text-stone-400" size={18} />
                <input 
                  required
                  type="number" 
                  value={newSale.price}
                  onChange={(e) => setNewSale({ ...newSale, price: Number(e.target.value) })}
                  className="w-full border border-stone-200 rounded-xl pr-10 pl-4 py-3 outline-none focus:ring-2 focus:ring-amber-500"
                />
              </div>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">איפה שמת את הכסף? (מיקום פיזי)</label>
              <div className="relative">
                <MapPin className="absolute right-3 top-1/2 -translate-y-1/2 text-stone-400" size={18} />
                <input 
                  type="text" 
                  value={newSale.moneyLocation}
                  onChange={(e) => setNewSale({ ...newSale, moneyLocation: e.target.value })}
                  placeholder="למשל: ארנק חום, מגירה שמאלית"
                  className="w-full border border-stone-200 rounded-xl pr-10 pl-4 py-3 outline-none focus:ring-2 focus:ring-amber-500"
                />
              </div>
            </div>
            <div className="space-y-1">
              <label className="text-sm font-bold text-stone-700">הערות</label>
              <input 
                type="text" 
                value={newSale.notes}
                onChange={(e) => setNewSale({ ...newSale, notes: e.target.value })}
                className="w-full border border-stone-200 rounded-xl px-4 py-3 outline-none focus:ring-2 focus:ring-amber-500"
              />
            </div>
            <div className="md:col-span-2 flex justify-end gap-3">
              <button 
                type="button"
                onClick={() => setIsAdding(false)}
                className="px-8 py-3 text-stone-500 font-bold"
              >
                ביטול
              </button>
              <button 
                type="submit"
                className="bg-amber-600 text-white px-10 py-3 rounded-xl font-bold shadow-lg hover:bg-amber-700 transition-all"
              >
                בצע רישום
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="bg-white rounded-3xl shadow-sm border border-stone-100 overflow-hidden">
        <table className="w-full text-right border-collapse">
          <thead>
            <tr className="bg-stone-50 border-b border-stone-100">
              <th className="px-6 py-4 text-sm font-bold text-stone-500">תאריך</th>
              <th className="px-6 py-4 text-sm font-bold text-stone-500">מוצר</th>
              <th className="px-6 py-4 text-sm font-bold text-stone-500">מיקום הכסף</th>
              <th className="px-6 py-4 text-sm font-bold text-stone-500">סכום</th>
              <th className="px-6 py-4 text-sm font-bold text-stone-500"></th>
            </tr>
          </thead>
          <tbody>
            {sales?.sort((a, b) => b.saleDate.getTime() - a.saleDate.getTime()).map((sale) => (
              <tr key={sale.id} className="border-b border-stone-50 hover:bg-stone-50/50 transition-colors">
                <td className="px-6 py-4">
                  <div className="flex items-center gap-2 text-sm text-stone-500">
                    <Calendar size={14} /> {format(sale.saleDate, 'dd/MM/yyyy HH:mm', { locale: he })}
                  </div>
                </td>
                <td className="px-6 py-4 font-bold text-stone-800">{sale.productName}</td>
                <td className="px-6 py-4 text-sm text-stone-600">
                  <span className="bg-stone-100 px-2 py-1 rounded-lg">{sale.moneyLocation || 'לא צוין'}</span>
                </td>
                <td className="px-6 py-4 font-black text-amber-600 text-lg">₪{sale.price}</td>
                <td className="px-6 py-4 text-left">
                  <button 
                    onClick={async () => {
                      if (confirm('האם למחוק את רישום המכירה?')) {
                        await db.sales.delete(sale.id!);
                      }
                    }}
                    className="text-stone-300 hover:text-red-500 transition-colors"
                  >
                    <Trash2 size={18} />
                  </button>
                </td>
              </tr>
            ))}
            {sales?.length === 0 && (
              <tr>
                <td colSpan={5} className="px-6 py-20 text-center text-stone-400 italic">
                  טרם בוצעו מכירות.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
    </div>
  );
}
