import React, { useState, useEffect } from 'react';
import { useLiveQuery } from 'dexie-react-hooks';
import { db } from './db';
import { 
  LayoutDashboard, 
  Package, 
  ShoppingCart, 
  Wrench, 
  Settings as SettingsIcon, 
  Database, 
  Info,
  ChevronRight,
  Plus,
  ArrowRightLeft,
  UserCheck,
  Store
} from 'lucide-react';
import { motion, AnimatePresence } from 'motion/react';

// Components (will be created in separate files)
import SetupWizard from './components/SetupWizard';
import OrdersView from './components/OrdersView';
import InventoryView from './components/InventoryView';
import ManagerView from './components/ManagerView';
import SellerView from './components/SellerView';
import CashierView from './components/CashierView';
import SyncView from './components/SyncView';
import TechnicalInfoView from './components/TechnicalInfoView';

export default function App() {
  const [activeTab, setActiveTab] = useState('dashboard');
  const [userRole, setUserRole] = useState<'manager' | 'seller' | null>(null);
  const settings = useLiveQuery(() => db.settings.toArray());
  const [isWizardComplete, setIsWizardComplete] = useState(false);

  useEffect(() => {
    if (settings && settings.length > 0) {
      const wizardStatus = settings.find(s => s.key === 'wizard_complete');
      const savedRole = settings.find(s => s.key === 'user_role');
      if (wizardStatus?.value) {
        setIsWizardComplete(true);
      }
      if (savedRole?.value) {
        setUserRole(savedRole.value);
      }
    }
  }, [settings]);

  // Notification Check
  useEffect(() => {
    const checkNotifications = async () => {
      const pendingOrders = await db.orders.where('status').equals('pending').toArray();
      const today = new Date();
      today.setHours(0, 0, 0, 0);

      const dueOrders = pendingOrders.filter(order => {
        const arrival = new Date(order.estimatedArrival);
        arrival.setHours(0, 0, 0, 0);
        return arrival <= today;
      });

      if (dueOrders.length > 0) {
        // Browser notification if supported and permitted
        if ("Notification" in window && Notification.permission === "granted") {
          new Notification("חבילות ממתינות לאיסוף!", {
            body: `ישנן ${dueOrders.length} חבילות שהגיעו או אמורות להגיע היום.`,
            icon: "/favicon.ico"
          });
        } else if ("Notification" in window && Notification.permission !== "denied") {
          Notification.requestPermission();
        }
      }
    };

    if (isWizardComplete) {
      checkNotifications();
    }
  }, [isWizardComplete]);

  const handleRoleSelect = async (role: 'manager' | 'seller') => {
    setUserRole(role);
    const existingRole = await db.settings.where('key').equals('user_role').first();
    if (existingRole) {
      await db.settings.update(existingRole.id!, { value: role });
    } else {
      await db.settings.add({ key: 'user_role', value: role });
    }
  };

  if (!isWizardComplete && settings !== undefined) {
    return <SetupWizard onComplete={() => setIsWizardComplete(true)} />;
  }

  if (!userRole && isWizardComplete) {
    return (
      <div className="min-h-screen bg-stone-100 flex items-center justify-center p-4">
        <div className="bg-white p-8 rounded-3xl shadow-xl max-w-md w-full text-center space-y-6">
          <h2 className="text-2xl font-bold">בחר את התפקיד שלך</h2>
          <p className="text-stone-500">התצוגה תותאם לפי הבחירה שלך.</p>
          <div className="grid grid-cols-2 gap-4">
            <button 
              onClick={() => handleRoleSelect('manager')}
              className="p-6 border-2 border-stone-100 rounded-2xl hover:border-emerald-500 hover:bg-emerald-50 transition-all group"
            >
              <UserCheck size={40} className="mx-auto mb-2 text-stone-400 group-hover:text-emerald-600" />
              <span className="font-bold">מנהל / מעבדה</span>
            </button>
            <button 
              onClick={() => handleRoleSelect('seller')}
              className="p-6 border-2 border-stone-100 rounded-2xl hover:border-blue-500 hover:bg-blue-50 transition-all group"
            >
              <Store size={40} className="mx-auto mb-2 text-stone-400 group-hover:text-blue-600" />
              <span className="font-bold">מוכר / שירות</span>
            </button>
          </div>
        </div>
      </div>
    );
  }

  const allTabs = [
    { id: 'dashboard', label: 'לוח בקרה', icon: LayoutDashboard, roles: ['manager', 'seller'] },
    { id: 'orders', label: 'הזמנות חו"ל', icon: Package, roles: ['manager', 'seller'] },
    { id: 'inventory', label: 'ניהול מלאי', icon: ShoppingCart, roles: ['manager'] },
    { id: 'technical', label: 'מידע טכני', icon: Info, roles: ['manager', 'seller'] },
    { id: 'manager', label: 'חלון מנהל', icon: UserCheck, roles: ['manager'] },
    { id: 'seller', label: 'חלון מוכר', icon: Store, roles: ['seller'] },
    { id: 'cashier', label: 'קופה', icon: Database, roles: ['manager', 'seller'] },
    { id: 'sync', label: 'סנכרון וגיבוי', icon: ArrowRightLeft, roles: ['manager', 'seller'] },
  ];

  const tabs = allTabs.filter(tab => tab.roles.includes(userRole!));

  return (
    <div className="min-h-screen bg-stone-50 text-stone-900 flex flex-col md:flex-row">
      {/* Sidebar */}
      <nav className="w-full md:w-64 bg-white border-l border-stone-200 p-4 flex flex-col gap-2 shadow-sm z-10">
        <div className="mb-8 px-2">
          <h1 className="text-xl font-bold text-emerald-700">ניהול לוגיסטי חכם</h1>
          <p className="text-xs text-stone-500">מערכת אופליין | {userRole === 'manager' ? 'מצב מנהל' : 'מצב מוכר'}</p>
          <button 
            onClick={() => handleRoleSelect(userRole === 'manager' ? 'seller' : 'manager')}
            className="mt-2 text-[10px] text-stone-400 hover:text-stone-600 underline"
          >
            החלף תפקיד
          </button>
        </div>
        
        {tabs.map((tab) => (
          <button
            key={tab.id}
            onClick={() => setActiveTab(tab.id)}
            className={`flex items-center gap-3 px-4 py-3 rounded-xl transition-all duration-200 ${
              activeTab === tab.id 
                ? 'bg-emerald-50 text-emerald-700 font-semibold shadow-sm' 
                : 'text-stone-600 hover:bg-stone-100'
            }`}
          >
            <tab.icon size={20} />
            <span>{tab.label}</span>
          </button>
        ))}
      </nav>

      {/* Main Content */}
      <main className="flex-1 p-4 md:p-8 overflow-auto">
        <AnimatePresence mode="wait">
          <motion.div
            key={activeTab}
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -10 }}
            transition={{ duration: 0.2 }}
          >
            {activeTab === 'dashboard' && <DashboardView setActiveTab={setActiveTab} />}
            {activeTab === 'orders' && <OrdersView />}
            {activeTab === 'inventory' && <InventoryView />}
            {activeTab === 'technical' && <TechnicalInfoView />}
            {activeTab === 'manager' && <ManagerView />}
            {activeTab === 'seller' && <SellerView />}
            {activeTab === 'cashier' && <CashierView />}
            {activeTab === 'sync' && <SyncView />}
          </motion.div>
        </AnimatePresence>
      </main>
    </div>
  );
}

function DashboardView({ setActiveTab }: { setActiveTab: (tab: string) => void }) {
  const orders = useLiveQuery(() => db.orders.where('status').equals('pending').toArray());
  const repairs = useLiveQuery(() => db.repairs.toArray());
  
  return (
    <div className="space-y-6">
      <header>
        <h2 className="text-3xl font-bold tracking-tight">שלום רב 👋</h2>
        <p className="text-stone-500">הנה תמונת מצב של העסק שלך להיום.</p>
      </header>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="bg-white p-6 rounded-2xl shadow-sm border border-stone-100 flex flex-col justify-between">
          <div>
            <p className="text-sm font-medium text-stone-500 uppercase tracking-wider">הזמנות בדרך</p>
            <h3 className="text-4xl font-bold mt-1 text-emerald-600">{orders?.length || 0}</h3>
          </div>
          <button 
            onClick={() => setActiveTab('orders')}
            className="mt-4 text-sm text-emerald-600 font-medium flex items-center gap-1 hover:underline"
          >
            לניהול הזמנות <ChevronRight size={16} />
          </button>
        </div>

        <div className="bg-white p-6 rounded-2xl shadow-sm border border-stone-100 flex flex-col justify-between">
          <div>
            <p className="text-sm font-medium text-stone-500 uppercase tracking-wider">מוצרים בטיפול</p>
            <h3 className="text-4xl font-bold mt-1 text-blue-600">{repairs?.filter(r => r.progress < 100).length || 0}</h3>
          </div>
          <button 
            onClick={() => setActiveTab('manager')}
            className="mt-4 text-sm text-blue-600 font-medium flex items-center gap-1 hover:underline"
          >
            לחלון מנהל <ChevronRight size={16} />
          </button>
        </div>

        <div className="bg-white p-6 rounded-2xl shadow-sm border border-stone-100 flex flex-col justify-between">
          <div>
            <p className="text-sm font-medium text-stone-500 uppercase tracking-wider">מכירות החודש</p>
            <h3 className="text-4xl font-bold mt-1 text-amber-600">₪0</h3>
          </div>
          <button 
            onClick={() => setActiveTab('cashier')}
            className="mt-4 text-sm text-amber-600 font-medium flex items-center gap-1 hover:underline"
          >
            לצפייה בדוחות <ChevronRight size={16} />
          </button>
        </div>
      </div>

      <div className="bg-emerald-600 text-white p-8 rounded-3xl shadow-lg relative overflow-hidden">
        <div className="relative z-10 max-w-lg">
          <h3 className="text-2xl font-bold mb-2">צריך להוסיף מוצר חדש?</h3>
          <p className="opacity-90 mb-6 text-lg">דווח על מוצר שהגיע מלקוח או עדכן את המלאי הקיים בקלות.</p>
          <div className="flex gap-4">
            <button 
              onClick={() => setActiveTab('seller')}
              className="bg-white text-emerald-700 px-6 py-3 rounded-xl font-bold shadow-md hover:bg-stone-50 transition-colors flex items-center gap-2"
            >
              <Plus size={20} /> קליטת מוצר חדש
            </button>
          </div>
        </div>
        <div className="absolute -bottom-12 -left-12 w-64 h-64 bg-white/10 rounded-full blur-3xl"></div>
      </div>
    </div>
  );
}
