import React, { useState, useEffect, useMemo } from 'react';
import { 
  Package, 
  ShoppingCart, 
  History, 
  Settings, 
  Plus, 
  Trash2, 
  CheckCircle2, 
  AlertCircle, 
  Clock, 
  Truck,
  ChevronRight,
  ChevronLeft,
  Search,
  X,
  Lock,
  LogOut,
  Camera,
  CreditCard,
  User,
  MapPin,
  Save
} from 'lucide-react';
import { motion, AnimatePresence } from 'motion/react';

// --- Types ---

enum ProductStatus {
  IN_STOCK = 'IN_STOCK',
  COMING_SOON = 'COMING_SOON',
  DISCONTINUED = 'DISCONTINUED',
  IN_TRANSIT = 'IN_TRANSIT'
}

interface Product {
  id: number;
  name: string;
  description: string;
  price: number;
  category: string;
  image_url: string;
  status: ProductStatus;
}

interface CartItem extends Product {
  quantity: number;
}

interface Sale {
  id: number;
  timestamp: string;
  buyer_name: string;
  payment_type: string;
  storage_location: string;
  total_amount: number;
  items: string;
}

enum OrderStatus {
  ORDERED = 'ORDERED',
  PICKED_UP = 'PICKED_UP',
  VERSION_EDITING = 'VERSION_EDITING',
  READY = 'READY'
}

interface Order {
  id: number;
  productName: string;
  orderDate: string;
  reminderDate: string;
  status: OrderStatus;
  versionStatus?: string;
  etaDays?: number;
  notes?: string;
}

// --- Components ---

const StatusBadge = ({ status }: { status: ProductStatus }) => {
  const configs = {
    [ProductStatus.IN_STOCK]: { label: 'יש במלאי', color: 'bg-emerald-100 text-emerald-700 border-emerald-200', icon: CheckCircle2 },
    [ProductStatus.COMING_SOON]: { label: 'נגמר - יבוא חדש', color: 'bg-amber-100 text-amber-700 border-amber-200', icon: Clock },
    [ProductStatus.DISCONTINUED]: { label: 'נגמר - לא יבוא', color: 'bg-rose-100 text-rose-700 border-rose-200', icon: AlertCircle },
    [ProductStatus.IN_TRANSIT]: { label: 'במשלוח', color: 'bg-blue-100 text-blue-700 border-blue-200', icon: Truck },
  };

  const config = configs[status];
  const Icon = config.icon;

  return (
    <span className={`inline-flex items-center gap-1 px-2 py-0.5 rounded-full text-xs font-medium border ${config.color}`}>
      <Icon size={12} />
      {config.label}
    </span>
  );
};

export default function App() {
  const [view, setView] = useState<'seller' | 'admin' | 'history' | 'orders' | 'reports'>('seller');
  const [products, setProducts] = useState<Product[]>([]);
  const [cart, setCart] = useState<CartItem[]>([]);
  const [sales, setSales] = useState<Sale[]>([]);
  const [orders, setOrders] = useState<Order[]>([]);
  const [isAdminAuthenticated, setIsAdminAuthenticated] = useState(false);
  const [adminPassword, setAdminPassword] = useState('');
  const [searchQuery, setSearchQuery] = useState('');
  const [isCheckoutModalOpen, setIsCheckoutModalOpen] = useState(false);
  
  // Form states
  const [checkoutForm, setCheckoutForm] = useState({
    buyer_name: '',
    payment_type: 'מזומן',
    storage_location: ''
  });

  const [editingProduct, setEditingProduct] = useState<Partial<Product> | null>(null);
  const [newOrderForm, setNewOrderForm] = useState({ productName: '', notes: '' });
  const [snoozeDays, setSnoozeDays] = useState<number>(1);
  
  const [isOnline, setIsOnline] = useState(navigator.onLine);
  
  // Load data from localStorage on mount
  useEffect(() => {
    const handleOnline = () => setIsOnline(true);
    const handleOffline = () => setIsOnline(false);
    window.addEventListener('online', handleOnline);
    window.addEventListener('offline', handleOffline);
    
    const savedProducts = localStorage.getItem('sales_app_products');
    if (savedProducts) {
      setProducts(JSON.parse(savedProducts));
    } else {
      const initialProducts: Product[] = [
        { id: 1, name: 'מוצר לדוגמה', description: 'תיאור קצר של המוצר', price: 50, category: 'כללי', image_url: '', status: ProductStatus.IN_STOCK }
      ];
      setProducts(initialProducts);
      localStorage.setItem('sales_app_products', JSON.stringify(initialProducts));
    }

    const savedSales = localStorage.getItem('sales_app_sales');
    if (savedSales) setSales(JSON.parse(savedSales));

    const savedOrders = localStorage.getItem('sales_app_orders');
    if (savedOrders) setOrders(JSON.parse(savedOrders));
  }, []);

  // Save data to localStorage
  useEffect(() => {
    if (products.length > 0) localStorage.setItem('sales_app_products', JSON.stringify(products));
  }, [products]);

  useEffect(() => {
    if (sales.length > 0) localStorage.setItem('sales_app_sales', JSON.stringify(sales));
  }, [sales]);

  useEffect(() => {
    localStorage.setItem('sales_app_orders', JSON.stringify(orders));
  }, [orders]);

  const handleAddToCart = (product: Product) => {
    if (product.status !== ProductStatus.IN_STOCK) return;
    setCart(prev => {
      const existing = prev.find(item => item.id === product.id);
      if (existing) {
        return prev.map(item => item.id === product.id ? { ...item, quantity: item.quantity + 1 } : item);
      }
      return [...prev, { ...product, quantity: 1 }];
    });
  };

  const removeFromCart = (productId: number) => {
    setCart(prev => prev.filter(item => item.id !== productId));
  };

  const totalCartAmount = useMemo(() => {
    return cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
  }, [cart]);

  const handleCheckout = (e: React.FormEvent) => {
    e.preventDefault();
    const newSale: Sale = {
      id: Date.now(),
      timestamp: new Date().toISOString(),
      buyer_name: checkoutForm.buyer_name,
      payment_type: checkoutForm.payment_type,
      storage_location: checkoutForm.storage_location,
      total_amount: totalCartAmount,
      items: cart.map(item => `${item.name} (x${item.quantity})`).join(', ')
    };
    const updatedSales = [newSale, ...sales];
    setSales(updatedSales);
    setCart([]);
    setIsCheckoutModalOpen(false);
    setCheckoutForm({ buyer_name: '', payment_type: 'מזומן', storage_location: '' });
    alert('המכירה תועדה בהצלחה!');
  };

  const handleSaveProduct = (e: React.FormEvent) => {
    e.preventDefault();
    if (!editingProduct) return;
    if (editingProduct.id) {
      setProducts(prev => prev.map(p => p.id === editingProduct.id ? (editingProduct as Product) : p));
    } else {
      setProducts(prev => [...prev, { ...editingProduct, id: Date.now() } as Product]);
    }
    setEditingProduct(null);
  };

  const handleDeleteProduct = (id: number) => {
    if (!confirm('האם אתה בטוח שברצונך למחוק מוצר זה?')) return;
    setProducts(prev => prev.filter(p => p.id !== id));
  };

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onloadend = () => setEditingProduct(prev => ({ ...prev, image_url: reader.result as string }));
      reader.readAsDataURL(file);
    }
  };

  const handleAddOrder = (e: React.FormEvent) => {
    e.preventDefault();
    const now = new Date();
    const reminder = new Date();
    reminder.setDate(now.getDate() + 14); // 2 weeks reminder

    const newOrder: Order = {
      id: Date.now(),
      productName: newOrderForm.productName,
      orderDate: now.toISOString(),
      reminderDate: reminder.toISOString(),
      status: OrderStatus.ORDERED,
      notes: newOrderForm.notes
    };
    setOrders(prev => [...prev, newOrder]);
    setNewOrderForm({ productName: '', notes: '' });
  };

  const updateOrderStatus = (id: number, status: OrderStatus, extra: Partial<Order> = {}) => {
    setOrders(prev => prev.map(o => o.id === id ? { ...o, status, ...extra } : o));
  };

  const snoozeReminder = (id: number, days: number) => {
    const newReminder = new Date();
    newReminder.setDate(newReminder.getDate() + days);
    setOrders(prev => prev.map(o => o.id === id ? { ...o, reminderDate: newReminder.toISOString() } : o));
  };

  const filteredProducts = products.filter(p => 
    p.name.toLowerCase().includes(searchQuery.toLowerCase()) || 
    p.category.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const handleAdminLogin = (e: React.FormEvent) => {
    e.preventDefault();
    if (adminPassword === '1234') {
      setIsAdminAuthenticated(true);
      setAdminPassword('');
    } else {
      alert('סיסמה שגויה');
    }
  };

  const activeReminders = orders.filter(o => 
    o.status === OrderStatus.ORDERED && new Date(o.reminderDate) <= new Date()
  );

  const versionEditingOrders = orders.filter(o => o.status === OrderStatus.VERSION_EDITING);

  return (
    <div className="min-h-screen flex flex-col font-sans text-slate-900">
      {/* Header */}
      <header className="sticky top-0 z-30 glass border-b border-slate-200 px-4 py-3 flex items-center justify-between">
        <div className="flex items-center gap-2">
          <div className="bg-indigo-600 p-2 rounded-lg text-white">
            <Package size={20} />
          </div>
          <div>
            <h1 className="text-xl font-bold tracking-tight leading-none">SalesManager</h1>
            <div className="flex items-center gap-1 mt-0.5">
              <div className={`w-1.5 h-1.5 rounded-full ${isOnline ? 'bg-emerald-500' : 'bg-rose-500'}`} />
              <span className="text-[10px] text-slate-500 font-medium">
                {isOnline ? 'מחובר' : 'מצב אופליין פעיל'}
              </span>
            </div>
          </div>
        </div>
        
        <nav className="flex items-center gap-1 bg-slate-100 p-1 rounded-xl">
          <button 
            onClick={() => setView('seller')}
            className={`px-4 py-1.5 rounded-lg text-sm font-medium transition-all ${view === 'seller' ? 'bg-white shadow-sm text-indigo-600' : 'text-slate-600 hover:text-slate-900'}`}
          >
            מכירה
          </button>
          <button 
            onClick={() => setView('admin')}
            className={`px-4 py-1.5 rounded-lg text-sm font-medium transition-all ${['admin', 'history', 'orders', 'reports'].includes(view) ? 'bg-white shadow-sm text-indigo-600' : 'text-slate-600 hover:text-slate-900'}`}
          >
            ניהול
          </button>
        </nav>
      </header>

      <main className="flex-1 max-w-5xl mx-auto w-full p-4 pb-24">
        {view === 'seller' && (
          <div className="space-y-6">
            {/* Incoming Inventory Alert for Seller */}
            {versionEditingOrders.length > 0 && (
              <div className="bg-indigo-50 border border-indigo-100 p-4 rounded-2xl">
                <h3 className="text-indigo-900 font-bold mb-2 flex items-center gap-2">
                  <Truck size={18} />
                  מוצרים בדרך (בעריכת גירסה):
                </h3>
                <div className="space-y-2">
                  {versionEditingOrders.map(o => (
                    <div key={o.id} className="text-sm text-indigo-700 flex justify-between">
                      <span>{o.productName} - {o.versionStatus}</span>
                      <span className="font-bold">צפוי בעוד {o.etaDays} ימים</span>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Search */}
            <div className="relative">
              <Search className="absolute right-3 top-1/2 -translate-y-1/2 text-slate-400" size={18} />
              <input 
                type="text"
                placeholder="חפש מוצר או קטגוריה..."
                className="w-full pr-10 pl-4 py-3 rounded-2xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition-all"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
            </div>

            {/* Product Grid */}
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
              {filteredProducts.map(product => (
                <motion.div 
                  layout
                  key={product.id}
                  className="bg-white rounded-2xl border border-slate-200 overflow-hidden shadow-sm hover:shadow-md transition-shadow flex flex-col"
                >
                  <div className="aspect-square relative bg-slate-100">
                    {product.image_url ? (
                      <img src={product.image_url} alt={product.name} className="w-full h-full object-cover" />
                    ) : (
                      <div className="w-full h-full flex items-center justify-center text-slate-300">
                        <Package size={48} />
                      </div>
                    )}
                    <div className="absolute top-2 right-2">
                      <StatusBadge status={product.status} />
                    </div>
                  </div>
                  <div className="p-3 flex-1 flex flex-col">
                    <div className="text-xs text-indigo-600 font-semibold mb-1 uppercase tracking-wider">{product.category}</div>
                    <h3 className="font-bold text-slate-900 mb-1 leading-tight">{product.name}</h3>
                    <p className="text-xs text-slate-500 line-clamp-2 mb-3 flex-1">{product.description}</p>
                    <div className="flex items-center justify-between mt-auto">
                      <span className="text-lg font-bold text-indigo-600">₪{product.price}</span>
                      <button 
                        onClick={() => handleAddToCart(product)}
                        disabled={product.status !== ProductStatus.IN_STOCK}
                        className={`p-2 rounded-xl transition-all ${product.status === ProductStatus.IN_STOCK ? 'bg-indigo-600 text-white hover:bg-indigo-700 active:scale-95' : 'bg-slate-100 text-slate-400 cursor-not-allowed'}`}
                      >
                        <Plus size={20} />
                      </button>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          </div>
        )}

        {view === 'admin' && !isAdminAuthenticated && (
          <div className="max-w-md mx-auto mt-20">
            <div className="bg-white p-8 rounded-3xl border border-slate-200 shadow-xl text-center">
              <div className="w-16 h-16 bg-indigo-100 text-indigo-600 rounded-2xl flex items-center justify-center mx-auto mb-6">
                <Lock size={32} />
              </div>
              <h2 className="text-2xl font-bold mb-2">אזור ניהול מוגן</h2>
              <p className="text-slate-500 mb-8 text-sm">אנא הזן את סיסמת המנהל כדי להמשיך</p>
              
              <form onSubmit={handleAdminLogin} className="space-y-4">
                <input 
                  type="password"
                  placeholder="סיסמה"
                  className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none text-center text-lg tracking-widest"
                  value={adminPassword}
                  onChange={(e) => setAdminPassword(e.target.value)}
                  autoFocus
                />
                <button 
                  type="submit"
                  className="w-full bg-indigo-600 text-white py-3 rounded-xl font-bold hover:bg-indigo-700 transition-all shadow-lg shadow-indigo-200"
                >
                  כניסה
                </button>
              </form>
            </div>
          </div>
        )}

        {['admin', 'history', 'orders', 'reports'].includes(view) && isAdminAuthenticated && (
          <div className="space-y-8">
            {/* Admin Sub-Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-slate-200 pb-4">
              <button onClick={() => setView('admin')} className={`px-4 py-2 rounded-xl text-sm font-bold ${view === 'admin' ? 'bg-indigo-600 text-white' : 'bg-white text-slate-600 border border-slate-200'}`}>מוצרים</button>
              <button onClick={() => setView('history')} className={`px-4 py-2 rounded-xl text-sm font-bold ${view === 'history' ? 'bg-indigo-600 text-white' : 'bg-white text-slate-600 border border-slate-200'}`}>מכירות</button>
              <button onClick={() => setView('orders')} className={`px-4 py-2 rounded-xl text-sm font-bold ${view === 'orders' ? 'bg-indigo-600 text-white' : 'bg-white text-slate-600 border border-slate-200'}`}>הזמנות {activeReminders.length > 0 && <span className="bg-rose-500 text-white px-1.5 rounded-full text-[10px]">{activeReminders.length}</span>}</button>
              <button onClick={() => setView('reports')} className={`px-4 py-2 rounded-xl text-sm font-bold ${view === 'reports' ? 'bg-indigo-600 text-white' : 'bg-white text-slate-600 border border-slate-200'}`}>דו"חות</button>
              <button onClick={() => setIsAdminAuthenticated(false)} className="mr-auto p-2 text-slate-400 hover:text-rose-600"><LogOut size={20} /></button>
            </div>

            {view === 'admin' && (
              <div className="space-y-6">
                <div className="flex items-center justify-between">
                  <h2 className="text-2xl font-bold">ניהול מוצרים ומלאי</h2>
                  <button 
                    onClick={() => setEditingProduct({ name: '', price: 0, category: '', description: '', status: ProductStatus.IN_STOCK })}
                    className="flex items-center gap-2 px-4 py-2 bg-indigo-600 text-white rounded-xl text-sm font-medium hover:bg-indigo-700"
                  >
                    <Plus size={18} />
                    מוצר חדש
                  </button>
                </div>

                <div className="bg-white rounded-2xl border border-slate-200 overflow-hidden">
                  <table className="w-full text-right">
                    <thead className="bg-slate-50 border-b border-slate-200">
                      <tr>
                        <th className="px-6 py-4 text-sm font-semibold text-slate-600">מוצר</th>
                        <th className="px-6 py-4 text-sm font-semibold text-slate-600">מחיר</th>
                        <th className="px-6 py-4 text-sm font-semibold text-slate-600">סטטוס מלאי</th>
                        <th className="px-6 py-4 text-sm font-semibold text-slate-600">פעולות</th>
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-slate-100">
                      {products.map(product => (
                        <tr key={product.id} className="hover:bg-slate-50/50 transition-colors">
                          <td className="px-6 py-4">
                            <div className="flex items-center gap-3">
                              <div className="w-10 h-10 rounded-lg bg-slate-100 overflow-hidden flex-shrink-0">
                                {product.image_url ? (
                                  <img src={product.image_url} alt="" className="w-full h-full object-cover" />
                                ) : (
                                  <div className="w-full h-full flex items-center justify-center text-slate-300">
                                    <Package size={20} />
                                  </div>
                                )}
                              </div>
                              <div>
                                <div className="font-bold text-slate-900">{product.name}</div>
                                <div className="text-xs text-slate-500">{product.category}</div>
                              </div>
                            </div>
                          </td>
                          <td className="px-6 py-4 font-semibold">₪{product.price}</td>
                          <td className="px-6 py-4">
                            <StatusBadge status={product.status} />
                          </td>
                          <td className="px-6 py-4">
                            <div className="flex gap-2">
                              <button onClick={() => setEditingProduct(product)} className="p-2 text-slate-400 hover:text-indigo-600"><Settings size={18} /></button>
                              <button onClick={() => handleDeleteProduct(product.id)} className="p-2 text-slate-400 hover:text-rose-600"><Trash2 size={18} /></button>
                            </div>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>
            )}

            {view === 'history' && (
              <div className="space-y-4">
                <h2 className="text-2xl font-bold">תיעוד מכירות</h2>
                {sales.map(sale => (
                  <div key={sale.id} className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
                    <div className="flex flex-wrap items-start justify-between gap-4 mb-4">
                      <div>
                        <div className="text-sm text-slate-500 mb-1">{new Date(sale.timestamp).toLocaleString('he-IL')}</div>
                        <h3 className="text-lg font-bold flex items-center gap-2"><User size={18} className="text-indigo-600" />{sale.buyer_name || 'לקוח לא רשום'}</h3>
                      </div>
                      <div className="text-left">
                        <div className="text-2xl font-bold text-indigo-600">₪{sale.total_amount}</div>
                        <div className="flex items-center gap-2 text-sm text-slate-500"><CreditCard size={14} />{sale.payment_type}</div>
                      </div>
                    </div>
                    <div className="border-t border-slate-100 pt-4 flex flex-wrap gap-x-8 gap-y-2 text-sm">
                      <div className="flex items-center gap-2"><Package size={16} className="text-slate-400" /><span className="text-slate-600 font-medium">מוצרים:</span><span>{sale.items}</span></div>
                      <div className="flex items-center gap-2"><MapPin size={16} className="text-slate-400" /><span className="text-slate-600 font-medium">מיקום תשלום:</span><span>{sale.storage_location}</span></div>
                    </div>
                  </div>
                ))}
              </div>
            )}

            {view === 'orders' && (
              <div className="space-y-8">
                <div className="flex items-center justify-between">
                  <h2 className="text-2xl font-bold">ניהול הזמנות ורכש</h2>
                </div>

                {/* New Order Form */}
                <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
                  <h3 className="font-bold mb-4">הזמנה חדשה מספק</h3>
                  <form onSubmit={handleAddOrder} className="flex flex-wrap gap-4">
                    <input 
                      required
                      type="text"
                      placeholder="שם המוצר המוזמן"
                      className="flex-1 min-w-[200px] px-4 py-2 rounded-xl border border-slate-200 outline-none"
                      value={newOrderForm.productName}
                      onChange={e => setNewOrderForm(prev => ({ ...prev, productName: e.target.value }))}
                    />
                    <input 
                      type="text"
                      placeholder="הערות (ספק, כמות וכו')"
                      className="flex-1 min-w-[200px] px-4 py-2 rounded-xl border border-slate-200 outline-none"
                      value={newOrderForm.notes}
                      onChange={e => setNewOrderForm(prev => ({ ...prev, notes: e.target.value }))}
                    />
                    <button type="submit" className="bg-indigo-600 text-white px-6 py-2 rounded-xl font-bold hover:bg-indigo-700">הזמן</button>
                  </form>
                </div>

                {/* Reminders Section */}
                {activeReminders.length > 0 && (
                  <div className="space-y-4">
                    <h3 className="text-rose-600 font-bold flex items-center gap-2"><AlertCircle size={18} /> התראות איסוף (עברו שבועיים):</h3>
                    {activeReminders.map(o => (
                      <div key={o.id} className="bg-rose-50 p-4 rounded-2xl border border-rose-100 flex flex-wrap items-center justify-between gap-4">
                        <div>
                          <div className="font-bold text-rose-900">{o.productName}</div>
                          <div className="text-xs text-rose-700">הוזמן ב: {new Date(o.orderDate).toLocaleDateString('he-IL')}</div>
                        </div>
                        <div className="flex items-center gap-2">
                          <button onClick={() => updateOrderStatus(o.id, OrderStatus.PICKED_UP)} className="bg-emerald-600 text-white px-4 py-1.5 rounded-lg text-sm font-bold">אספתי</button>
                          <div className="flex items-center gap-1 bg-white p-1 rounded-lg border border-rose-200">
                            <input type="number" className="w-12 text-center outline-none" value={snoozeDays} onChange={e => setSnoozeDays(Number(e.target.value))} min="1" />
                            <button onClick={() => snoozeReminder(o.id, snoozeDays)} className="text-rose-600 text-xs font-bold px-2">הזכר לי שוב</button>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}

                {/* Version Editing Section */}
                <div className="space-y-4">
                  <h3 className="font-bold flex items-center gap-2"><Settings size={18} /> מכשירים בעריכת גירסה:</h3>
                  {orders.filter(o => [OrderStatus.PICKED_UP, OrderStatus.VERSION_EDITING].includes(o.status)).map(o => (
                    <div key={o.id} className="bg-white p-4 rounded-2xl border border-slate-200 shadow-sm space-y-3">
                      <div className="flex justify-between items-start">
                        <div className="font-bold">{o.productName}</div>
                        <div className="text-xs bg-slate-100 px-2 py-1 rounded-full">{o.status === OrderStatus.PICKED_UP ? 'מחכה לעריכה' : 'בעריכה'}</div>
                      </div>
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <input 
                          type="text" 
                          placeholder="מצב נוכחי (למשל: צריבת תוכנה)" 
                          className="px-3 py-1.5 rounded-lg border border-slate-200 text-sm outline-none"
                          value={o.versionStatus || ''}
                          onChange={e => updateOrderStatus(o.id, OrderStatus.VERSION_EDITING, { versionStatus: e.target.value })}
                        />
                        <input 
                          type="number" 
                          placeholder="ימים לסיום" 
                          className="px-3 py-1.5 rounded-lg border border-slate-200 text-sm outline-none"
                          value={o.etaDays || ''}
                          onChange={e => updateOrderStatus(o.id, OrderStatus.VERSION_EDITING, { etaDays: Number(e.target.value) })}
                        />
                        <button 
                          onClick={() => {
                            updateOrderStatus(o.id, OrderStatus.READY);
                            alert('המוצר מוכן! אל תשכח לעדכן אותו במלאי המוצרים.');
                          }} 
                          className="bg-indigo-600 text-white px-4 py-1.5 rounded-lg text-sm font-bold"
                        >
                          סיים עריכה (מוכן)
                        </button>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {view === 'reports' && (
              <div className="space-y-6">
                <h2 className="text-2xl font-bold">דו"חות וסיכומים</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm text-center">
                    <div className="text-slate-500 text-sm mb-1">סה"כ מכירות</div>
                    <div className="text-3xl font-black text-indigo-600">₪{sales.reduce((s, v) => s + v.total_amount, 0)}</div>
                  </div>
                  <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm text-center">
                    <div className="text-slate-500 text-sm mb-1">כמות עסקאות</div>
                    <div className="text-3xl font-black text-indigo-600">{sales.length}</div>
                  </div>
                  <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm text-center">
                    <div className="text-slate-500 text-sm mb-1">מוצרים במלאי</div>
                    <div className="text-3xl font-black text-indigo-600">{products.filter(p => p.status === ProductStatus.IN_STOCK).length}</div>
                  </div>
                </div>

                <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
                  <h3 className="font-bold mb-4">פירוט לפי אמצעי תשלום</h3>
                  <div className="space-y-2">
                    {['מזומן', 'אשראי', 'העברה בנקאית', 'ביט / פייבוקס'].map(type => {
                      const amount = sales.filter(s => s.payment_type === type).reduce((s, v) => s + v.total_amount, 0);
                      return (
                        <div key={type} className="flex justify-between items-center py-2 border-b border-slate-50">
                          <span>{type}</span>
                          <span className="font-bold">₪{amount}</span>
                        </div>
                      );
                    })}
                  </div>
                </div>
              </div>
            )}
          </div>
        )}
      </main>

      {/* Cart Footer Bar */}
      {view === 'seller' && cart.length > 0 && (
        <div className="fixed bottom-0 left-0 right-0 p-4 z-40">
          <motion.div 
            initial={{ y: 100 }}
            animate={{ y: 0 }}
            className="max-w-md mx-auto bg-indigo-600 text-white rounded-2xl shadow-2xl p-4 flex items-center justify-between"
          >
            <div className="flex items-center gap-3">
              <div className="bg-white/20 p-2 rounded-lg relative">
                <ShoppingCart size={20} />
                <span className="absolute -top-2 -right-2 bg-rose-500 text-white text-[10px] font-bold w-5 h-5 rounded-full flex items-center justify-center border-2 border-indigo-600">
                  {cart.reduce((s, i) => s + i.quantity, 0)}
                </span>
              </div>
              <div>
                <div className="text-xs opacity-80">סה"כ לתשלום</div>
                <div className="text-xl font-bold leading-none">₪{totalCartAmount}</div>
              </div>
            </div>
            <button 
              onClick={() => setIsCheckoutModalOpen(true)}
              className="bg-white text-indigo-600 px-6 py-2.5 rounded-xl font-bold hover:bg-slate-50 transition-all active:scale-95"
            >
              סיכום ותשלום
            </button>
          </motion.div>
        </div>
      )}

      {/* Checkout Modal */}
      <AnimatePresence>
        {isCheckoutModalOpen && (
          <div className="fixed inset-0 z-50 flex items-end sm:items-center justify-center p-4">
            <motion.div 
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              onClick={() => setIsCheckoutModalOpen(false)}
              className="absolute inset-0 bg-slate-900/60 backdrop-blur-sm"
            />
            <motion.div 
              initial={{ y: '100%' }}
              animate={{ y: 0 }}
              exit={{ y: '100%' }}
              className="relative w-full max-w-lg bg-white rounded-t-[32px] sm:rounded-[32px] shadow-2xl overflow-hidden"
            >
              <div className="p-6 overflow-y-auto max-h-[80vh]">
                <div className="flex items-center justify-between mb-6">
                  <h2 className="text-2xl font-bold">סיכום מכירה</h2>
                  <button onClick={() => setIsCheckoutModalOpen(false)} className="p-2 hover:bg-slate-100 rounded-full">
                    <X size={24} />
                  </button>
                </div>

                <div className="space-y-4 mb-8">
                  {cart.map(item => (
                    <div key={item.id} className="flex items-center justify-between bg-slate-50 p-3 rounded-xl">
                      <div className="flex items-center gap-3">
                        <div className="w-12 h-12 rounded-lg bg-white border border-slate-200 overflow-hidden">
                          {item.image_url && <img src={item.image_url} alt="" className="w-full h-full object-cover" />}
                        </div>
                        <div>
                          <div className="font-bold">{item.name}</div>
                          <div className="text-xs text-slate-500">₪{item.price} × {item.quantity}</div>
                        </div>
                      </div>
                      <div className="flex items-center gap-3">
                        <span className="font-bold">₪{item.price * item.quantity}</span>
                        <button onClick={() => removeFromCart(item.id)} className="text-rose-500 p-1">
                          <Trash2 size={16} />
                        </button>
                      </div>
                    </div>
                  ))}
                  <div className="flex items-center justify-between pt-4 border-t border-slate-100">
                    <span className="text-lg font-bold">סה"כ</span>
                    <span className="text-2xl font-black text-indigo-600">₪{totalCartAmount}</span>
                  </div>
                </div>

                <form onSubmit={handleCheckout} className="space-y-4">
                  <div>
                    <label className="block text-sm font-semibold mb-1.5 text-slate-700">שם הקונה</label>
                    <input 
                      required
                      type="text"
                      className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none"
                      value={checkoutForm.buyer_name}
                      onChange={(e) => setCheckoutForm(prev => ({ ...prev, buyer_name: e.target.value }))}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-semibold mb-1.5 text-slate-700">סוג תשלום</label>
                      <select 
                        className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none appearance-none bg-white"
                        value={checkoutForm.payment_type}
                        onChange={(e) => setCheckoutForm(prev => ({ ...prev, payment_type: e.target.value }))}
                      >
                        <option>מזומן</option>
                        <option>אשראי</option>
                        <option>העברה בנקאית</option>
                        <option>ביט / פייבוקס</option>
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-semibold mb-1.5 text-slate-700">איפה נשמר הכסף?</label>
                      <input 
                        required
                        placeholder="למשל: ארנק, מגירה..."
                        type="text"
                        className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none"
                        value={checkoutForm.storage_location}
                        onChange={(e) => setCheckoutForm(prev => ({ ...prev, storage_location: e.target.value }))}
                      />
                    </div>
                  </div>
                  <button 
                    type="submit"
                    className="w-full bg-indigo-600 text-white py-4 rounded-2xl font-bold text-lg hover:bg-indigo-700 transition-all shadow-xl shadow-indigo-100 mt-4"
                  >
                    אישור וסיום מכירה
                  </button>
                </form>
              </div>
            </motion.div>
          </div>
        )}
      </AnimatePresence>

      {/* Product Edit Modal */}
      <AnimatePresence>
        {editingProduct && (
          <div className="fixed inset-0 z-50 flex items-center justify-center p-4">
            <motion.div 
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              onClick={() => setEditingProduct(null)}
              className="absolute inset-0 bg-slate-900/60 backdrop-blur-sm"
            />
            <motion.div 
              initial={{ scale: 0.9, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              exit={{ scale: 0.9, opacity: 0 }}
              className="relative w-full max-w-2xl bg-white rounded-[32px] shadow-2xl overflow-hidden"
            >
              <form onSubmit={handleSaveProduct} className="p-8">
                <div className="flex items-center justify-between mb-8">
                  <h2 className="text-2xl font-bold">{editingProduct.id ? 'עריכת מוצר' : 'מוצר חדש'}</h2>
                  <button type="button" onClick={() => setEditingProduct(null)} className="p-2 hover:bg-slate-100 rounded-full">
                    <X size={24} />
                  </button>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-4">
                    <div className="aspect-square bg-slate-50 rounded-3xl border-2 border-dashed border-slate-200 relative overflow-hidden group">
                      {editingProduct.image_url ? (
                        <>
                          <img src={editingProduct.image_url} alt="" className="w-full h-full object-cover" />
                          <button 
                            type="button"
                            onClick={() => setEditingProduct(prev => ({ ...prev, image_url: '' }))}
                            className="absolute top-2 right-2 p-2 bg-rose-500 text-white rounded-xl opacity-0 group-hover:opacity-100 transition-opacity"
                          >
                            <Trash2 size={16} />
                          </button>
                        </>
                      ) : (
                        <label className="w-full h-full flex flex-col items-center justify-center cursor-pointer hover:bg-slate-100 transition-colors">
                          <Camera size={40} className="text-slate-300 mb-2" />
                          <span className="text-sm font-medium text-slate-500">הוסף תמונה</span>
                          <input type="file" accept="image/*" className="hidden" onChange={handleImageUpload} />
                        </label>
                      )}
                    </div>
                  </div>

                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-semibold mb-1.5 text-slate-700">שם המוצר</label>
                      <input 
                        required
                        type="text"
                        className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none"
                        value={editingProduct.name}
                        onChange={(e) => setEditingProduct(prev => ({ ...prev, name: e.target.value }))}
                      />
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-semibold mb-1.5 text-slate-700">מחיר (₪)</label>
                        <input 
                          required
                          type="number"
                          className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none"
                          value={editingProduct.price}
                          onChange={(e) => setEditingProduct(prev => ({ ...prev, price: Number(e.target.value) }))}
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-semibold mb-1.5 text-slate-700">קטגוריה</label>
                        <input 
                          required
                          type="text"
                          className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none"
                          value={editingProduct.category}
                          onChange={(e) => setEditingProduct(prev => ({ ...prev, category: e.target.value }))}
                        />
                      </div>
                    </div>
                    <div>
                      <label className="block text-sm font-semibold mb-1.5 text-slate-700">סטטוס מלאי</label>
                      <select 
                        className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none appearance-none bg-white"
                        value={editingProduct.status}
                        onChange={(e) => setEditingProduct(prev => ({ ...prev, status: e.target.value as ProductStatus }))}
                      >
                        <option value={ProductStatus.IN_STOCK}>יש במלאי</option>
                        <option value={ProductStatus.COMING_SOON}>נגמר - יבוא חדש</option>
                        <option value={ProductStatus.DISCONTINUED}>נגמר - לא יבוא יותר</option>
                        <option value={ProductStatus.IN_TRANSIT}>במשלוח - יגיע בקרוב</option>
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-semibold mb-1.5 text-slate-700">תיאור המוצר</label>
                      <textarea 
                        rows={3}
                        className="w-full px-4 py-3 rounded-xl border border-slate-200 focus:ring-2 focus:ring-indigo-500 outline-none resize-none"
                        value={editingProduct.description}
                        onChange={(e) => setEditingProduct(prev => ({ ...prev, description: e.target.value }))}
                      />
                    </div>
                  </div>
                </div>

                <div className="mt-10 flex gap-4">
                  <button 
                    type="submit"
                    className="flex-1 bg-indigo-600 text-white py-4 rounded-2xl font-bold text-lg hover:bg-indigo-700 transition-all flex items-center justify-center gap-2"
                  >
                    <Save size={20} />
                    שמור מוצר
                  </button>
                  <button 
                    type="button"
                    onClick={() => setEditingProduct(null)}
                    className="px-8 py-4 rounded-2xl font-bold text-lg border border-slate-200 hover:bg-slate-50 transition-all"
                  >
                    ביטול
                  </button>
                </div>
              </form>
            </motion.div>
          </div>
        )}
      </AnimatePresence>
    </div>
  );
}
