// Function to create and show the video player overlay
function showVideoPlayer(videos, initialIndex) {
    const existingPlayer = document.getElementById('netfri-video-player-overlay');
    if (existingPlayer) {
        existingPlayer.remove();
    }

    const overlay = document.createElement('div');
    overlay.id = 'netfri-video-player-overlay';
    overlay.style.position = 'fixed';
    overlay.style.top = '0';
    overlay.style.left = '0';
    overlay.style.width = '100%';
    overlay.style.height = '100%';
    overlay.style.backgroundColor = 'rgba(0, 0, 0, 0.9)';
    overlay.style.zIndex = '9999';
    overlay.style.display = 'flex';
    overlay.style.flexDirection = 'column';
    overlay.style.justifyContent = 'center';
    overlay.style.alignItems = 'center';

    const titleElement = document.createElement('div');
    titleElement.className = 'netfri-player-title';

    const playerContainer = document.createElement('div');
    playerContainer.className = 'netfri-player-container';

    const videoPlayer = document.createElement('video');
    videoPlayer.controls = true;
    videoPlayer.autoplay = true;
    videoPlayer.style.maxWidth = '100%';
    videoPlayer.style.maxHeight = '100%';
    videoPlayer.style.border = '3px solid #17a2b8';
    videoPlayer.style.borderRadius = '8px';
    videoPlayer.style.display = 'block';

    const prevButton = document.createElement('div');
    prevButton.className = 'netfri-player-button netfri-nav-prev';
    prevButton.innerHTML = `<svg viewBox="0 0 24 24"><path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path></svg>`;

    const nextButton = document.createElement('div');
    nextButton.className = 'netfri-player-button netfri-nav-next';
    nextButton.innerHTML = `<svg viewBox="0 0 24 24"><path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path></svg>`;

    const closeButton = document.createElement('div');
    closeButton.className = 'netfri-player-button netfri-close-button';
    closeButton.innerHTML = `<svg viewBox="0 0 24 24"><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"></path></svg>`;
    
    let handleKeyDown;
    const closePlayer = () => {
        overlay.remove();
        window.removeEventListener('keydown', handleKeyDown);
    };
    handleKeyDown = (event) => {
        if (event.key === 'ArrowLeft') { prevButton.click(); } 
        else if (event.key === 'ArrowRight') { nextButton.click(); } 
        else if (event.key === 'Escape') { closePlayer(); }
    };
    window.addEventListener('keydown', handleKeyDown);
    closeButton.addEventListener('click', closePlayer);
    
    playerContainer.appendChild(videoPlayer);
    playerContainer.appendChild(prevButton);
    playerContainer.appendChild(nextButton);
    playerContainer.appendChild(closeButton);
    
    overlay.appendChild(titleElement);
    overlay.appendChild(playerContainer);
    document.body.appendChild(overlay);

    let currentIndex = initialIndex;
    function loadVideo(index) {
        const video = videos[index];
        videoPlayer.src = video.url;
        const titleText = document.createElement('span');
        titleText.textContent = video.title;
        const countText = document.createElement('span');
        countText.className = 'netfri-player-title-count';
        countText.textContent = ` (${index + 1}/${videos.length})`;
        titleElement.innerHTML = '';
        titleElement.appendChild(titleText);
        titleElement.appendChild(countText);
        prevButton.style.display = (index === videos.length - 1) ? 'none' : 'block';
        nextButton.style.display = (index === 0) ? 'none' : 'block';
    }

    nextButton.addEventListener('click', (e) => { e.stopPropagation(); if (currentIndex > 0) { currentIndex--; loadVideo(currentIndex); } });
    prevButton.addEventListener('click', (e) => { e.stopPropagation(); if (currentIndex < videos.length - 1) { currentIndex++; loadVideo(currentIndex); } });
    
    playerContainer.addEventListener('click', (event) => {
        if (event.target === playerContainer) {
            closePlayer();
        }
    });
    overlay.addEventListener('click', (event) => {
        if (event.target === overlay) {
            closePlayer();
        }
    });

    loadVideo(initialIndex);
}


function addPlayButtons() {
    const videoExtensions = ['.mp4', '.mov', '.avi', '.wmv', '.webm'];
    
    // We need to rebuild the list every time to get the correct order on click
    const buildVideosList = () => {
        const currentVideosList = [];
        document.querySelectorAll('.netfri-attachment-container').forEach(container => {
            const videoLink = container.querySelector('a[href*="&disp=safe"]');
            if (videoLink) {
                currentVideosList.push({
                    url: videoLink.href,
                    title: container.getAttribute('data-filename')
                });
            }
        });
        return currentVideosList;
    };

    const allNameSpans = document.querySelectorAll('span.aV3');
    allNameSpans.forEach(span => {
        const filename = span.textContent.trim();
        const isVideo = videoExtensions.some(ext => filename.toLowerCase().endsWith(ext));
        
        if (isVideo) {
            const linkElement = span.closest('a[href*="&disp=safe"]');
            const container = linkElement ? linkElement.parentElement : null;

            if (!container || container.classList.contains('netfri-attachment-container')) {
                return;
            }

            // --- THE CORRECT, SAFE LOGIC ---
            container.classList.add('netfri-attachment-container');
            container.setAttribute('data-filename', filename);

            const overlay = document.createElement('div');
            overlay.className = 'netfri-play-overlay';
            overlay.innerHTML = `<div class="netfri-play-icon"><svg viewBox="0 0 24 24" fill="white" stroke-width="1.5" stroke="white"><polygon points="5 3 19 12 5 21 5 3"></polygon></svg></div>`;
            
            overlay.addEventListener('click', (event) => {
                event.preventDefault();
                event.stopPropagation();
                const videosList = buildVideosList();
                const currentIndex = videosList.findIndex(v => v.url === linkElement.href);
                showVideoPlayer(videosList, currentIndex);
            });

            const titleBelow = document.createElement('div');
            titleBelow.className = 'netfri-title-below';
            titleBelow.textContent = filename;

            container.appendChild(overlay);
            container.appendChild(titleBelow);
        }
    });
}

const observer = new MutationObserver(() => {
    addPlayButtons();
});
observer.observe(document.body, { childList: true, subtree: true });