/* Sticky Site Notes - popup */
(() => {
  'use strict';

  const STORAGE_KEY = 'ssn_notes_v1';

  const $ = (id) => document.getElementById(id);

  const elCurrentHost = $('currentHost');
  const elBtnAdd = $('btnAdd');
  const elList = $('list');
  const elSearch = $('search');
  const elCountAll = $('countAll');
  const elEmpty = $('empty');
  const elSiteNotes = $('siteNotes');

  let activeTab = null;
  let currentHost = '';
  let allNotes = []; // flattened notes

  function safeUrl(url) {
    try { return new URL(url); } catch { return null; }
  }

  function fmtDate(ts) {
    try {
      const d = new Date(ts);
      return d.toLocaleString('he-IL', { year:'numeric', month:'2-digit', day:'2-digit', hour:'2-digit', minute:'2-digit' });
    } catch {
      return '';
    }
  }

  function snippet(text) {
    const t = String(text || '').trim().replace(/\s+/g, ' ');
    if (!t) return '— (פתק ריק)';
    return t.length > 120 ? t.slice(0, 120) + '…' : t;
  }

  async function storageGetAll() {
    return await new Promise((resolve) => {
      chrome.storage.local.get([STORAGE_KEY], (res) => {
        resolve(res?.[STORAGE_KEY] || {});
      });
    });
  }

  async function storageSetAll(all) {
    return await new Promise((resolve) => {
      chrome.storage.local.set({ [STORAGE_KEY]: all }, () => resolve());
    });
  }

  function flatten(all) {
    const out = [];
    for (const [host, arr] of Object.entries(all || {})) {
      if (!Array.isArray(arr)) continue;
      for (const n of arr) {
        out.push({ host, ...n });
      }
    }
    // newest first
    out.sort((a, b) => (b.updatedAt || 0) - (a.updatedAt || 0));
    return out;
  }

  async function refresh() {
    const all = await storageGetAll();
    allNotes = flatten(all);
    elCountAll.textContent = String(allNotes.length);

    renderCurrentSite(all);
    renderList();
  }

  function renderCurrentSite(all) {
    elSiteNotes.innerHTML = '';
    if (!currentHost) return;
    const arr = Array.isArray(all?.[currentHost]) ? all[currentHost] : [];
    if (arr.length === 0) {
      const div = document.createElement('div');
      div.className = 'muted small';
      div.textContent = 'אין עדיין פתקים לאתר הזה.';
      elSiteNotes.appendChild(div);
      return;
    }

    for (const n of arr.slice().sort((a,b)=> (b.updatedAt||0)-(a.updatedAt||0))) {
      const row = document.createElement('div');
      row.className = 'siteNoteRow';

      const txt = document.createElement('div');
      txt.className = 'txt';
      txt.textContent = snippet(n.text);

      const actions = document.createElement('div');
      actions.className = 'actions';

      const btnFocus = document.createElement('button');
      btnFocus.className = 'iconbtn';
      btnFocus.title = 'הדגש באתר';
      btnFocus.textContent = '🎯';
      btnFocus.addEventListener('click', async () => {
        if (!activeTab?.id) return;
        try {
          await chrome.tabs.sendMessage(activeTab.id, { type: 'SSN_FOCUS_NOTE', id: String(n.id) });
        } catch {}
      });

      const btnDelete = document.createElement('button');
      btnDelete.className = 'iconbtn';
      btnDelete.title = 'מחק';
      btnDelete.textContent = '🗑';
      btnDelete.addEventListener('click', async () => {
        await deleteNote(currentHost, String(n.id));
      });

      actions.appendChild(btnFocus);
      actions.appendChild(btnDelete);

      row.appendChild(txt);
      row.appendChild(actions);
      elSiteNotes.appendChild(row);
    }
  }

  function renderList() {
    const q = String(elSearch.value || '').trim().toLowerCase();
    const filtered = q
      ? allNotes.filter(n => (String(n.host||'').toLowerCase().includes(q) || String(n.text||'').toLowerCase().includes(q)))
      : allNotes;

    elList.innerHTML = '';
    elEmpty.style.display = filtered.length ? 'none' : 'block';

    for (const n of filtered) {
      const item = document.createElement('div');
      item.className = 'item';

      const topline = document.createElement('div');
      topline.className = 'topline';

      const host = document.createElement('div');
      host.className = 'host';
      host.textContent = n.host || '—';

      const actions = document.createElement('div');
      actions.className = 'actions';

      const btnDelete = document.createElement('button');
      btnDelete.className = 'iconbtn';
      btnDelete.title = 'מחק פתק';
      btnDelete.textContent = '🗑';
      btnDelete.addEventListener('click', async () => {
        await deleteNote(String(n.host), String(n.id));
      });

      actions.appendChild(btnDelete);

      topline.appendChild(host);
      topline.appendChild(actions);

      const sn = document.createElement('div');
      sn.className = 'snippet';
      sn.textContent = snippet(n.text);

      const meta = document.createElement('div');
      meta.className = 'meta';
      meta.innerHTML = `<span>עודכן: ${fmtDate(n.updatedAt || n.createdAt || 0)}</span><span>${n.origin ? n.origin : ''}</span>`;

      item.appendChild(topline);
      item.appendChild(sn);
      item.appendChild(meta);

      elList.appendChild(item);
    }
  }

  async function deleteNote(host, id) {
    const all = await storageGetAll();
    const arr = Array.isArray(all?.[host]) ? all[host] : [];
    const next = arr.filter(n => String(n.id) !== String(id));
    all[host] = next;
    await storageSetAll(all);

    // remove live note in current tab (best effort)
    if (activeTab?.id && host === currentHost) {
      try { await chrome.tabs.sendMessage(activeTab.id, { type: 'SSN_DELETE_NOTE', id: String(id) }); } catch {}
    }

    await refresh();
  }

  async function init() {
    // active tab
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    activeTab = tabs?.[0] || null;

    // current host
    const u = safeUrl(activeTab?.url || '');
    // On chrome:// pages content script doesn't run; handle gracefully
    currentHost = (u && u.hostname) ? u.hostname.toLowerCase() : '';
    elCurrentHost.textContent = currentHost ? currentHost : 'לא זמין (עמוד פנימי של הדפדפן)';

    elBtnAdd.disabled = !currentHost;
    elBtnAdd.addEventListener('click', async () => {
      if (!activeTab?.id || !currentHost) return;
      try {
        await chrome.tabs.sendMessage(activeTab.id, { type: 'SSN_CREATE_NOTE' });
      } catch {
        // content script may not be available (e.g., chrome:// or blocked)
      }
      // update UI
      await refresh();
    });

    elSearch.addEventListener('input', renderList);

    await refresh();
  }

  init().catch(() => {});
})();
