/* Sticky Site Notes - content script (MV3)
   Saves notes per hostname in chrome.storage.local
*/
(() => {
  'use strict';

  const STORAGE_KEY = 'ssn_notes_v1';
  const UI_NS = 'ssn';
  const Z = 2147483647; // max-ish z-index

  const DEFAULT_NOTE = {
    text: '',
    x: 24,
    y: 24,
    w: 320,
    h: 220,
    color: '#FFF5A6',
    minimized: false,
    createdAt: 0,
    updatedAt: 0,
    origin: ''
  };

  const COLORS = [
    '#FFF5A6', // yellow
    '#FFD7E8', // pink
    '#D7F5FF', // light blue
    '#DFFFD6', // light green
    '#E9E2FF', // lavender
    '#FFFFFF', // white
    '#1E1E1E'  // dark
  ];

  const state = {
    host: safeHost(location.hostname),
    origin: safeOrigin(location.href),
    notes: new Map(), // id -> noteData
    els: new Map(),   // id -> {root, textarea, ...}
    saveTimers: new Map()
  };

  function safeHost(host) {
    try { return (host || '').toLowerCase(); } catch { return ''; }
  }
  function safeOrigin(href) {
    try { return new URL(href).origin; } catch { return ''; }
  }
  function now() { return Date.now(); }
  function uid() {
    // simple unique id without crypto requirements
    return 'n_' + Math.random().toString(36).slice(2, 10) + '_' + Date.now().toString(36);
  }

  function clamp(n, min, max) { return Math.max(min, Math.min(max, n)); }

  function cssEscape(s) {
    // minimal escape for querySelector
    return s.replace(/[^a-zA-Z0-9_-]/g, (m) => '\\' + m);
  }

  function ensureStyles() {
    if (document.getElementById(`${UI_NS}-style`)) return;
    const style = document.createElement('style');
    style.id = `${UI_NS}-style`;
    style.textContent = `
      .${UI_NS}-note{
        position: fixed;
        left: 0; top: 0;
        z-index: ${Z};
        width: 320px; height: 220px;
        border: 1px solid rgba(0,0,0,.18);
        border-radius: 12px;
        box-shadow: 0 16px 40px rgba(0,0,0,.25);
        display: flex;
        flex-direction: column;
        overflow: hidden;
        transform: translate3d(24px,24px,0);
        animation: ${UI_NS}-pop .22s ease-out;
        font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
      }
      @keyframes ${UI_NS}-pop{
        from { transform: translate3d(var(--x,24px), var(--y,24px), 0) scale(.96); opacity: .0; }
        to   { transform: translate3d(var(--x,24px), var(--y,24px), 0) scale(1); opacity: 1; }
      }
      .${UI_NS}-header{
        height: 34px;
        flex: 0 0 auto;
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 0 8px 0 10px;
        cursor: grab;
        user-select: none;
        border-bottom: 1px solid rgba(0,0,0,.12);
        background: rgba(255,255,255,.35);
        backdrop-filter: blur(8px);
      }
      .${UI_NS}-title{
        font-size: 12px;
        font-weight: 700;
        opacity: .85;
        display: flex;
        gap: 8px;
        align-items: center;
        min-width: 0;
      }
      .${UI_NS}-badge{
        font-size: 10px;
        opacity: .7;
        font-weight: 600;
        padding: 2px 6px;
        border: 1px solid rgba(0,0,0,.12);
        border-radius: 999px;
        white-space: nowrap;
      }
      .${UI_NS}-btns{
        display: flex;
        gap: 6px;
        align-items: center;
      }
      .${UI_NS}-iconbtn{
        width: 24px; height: 24px;
        border-radius: 8px;
        border: 1px solid rgba(0,0,0,.12);
        background: rgba(255,255,255,.65);
        cursor: pointer;
        display: grid;
        place-items: center;
        font-size: 14px;
        line-height: 1;
      }
      .${UI_NS}-iconbtn:hover{ filter: brightness(1.03); }
      .${UI_NS}-iconbtn:active{ transform: translateY(1px); }
      .${UI_NS}-body{
        flex: 1 1 auto;
        display: flex;
        background: transparent;
      }
      .${UI_NS}-text{
        width: 100%;
        height: 100%;
        border: 0;
        outline: 0;
        resize: none;
        padding: 10px 10px 12px;
        font-size: 14px;
        background: transparent;
        color: #111;
      }
      .${UI_NS}-note.${UI_NS}-dark .${UI_NS}-header{ background: rgba(0,0,0,.25); border-bottom-color: rgba(255,255,255,.16); }
      .${UI_NS}-note.${UI_NS}-dark .${UI_NS}-text{ color: #f5f5f5; }
      .${UI_NS}-note.${UI_NS}-dark .${UI_NS}-badge{ border-color: rgba(255,255,255,.2); }
      .${UI_NS}-palette{
        position: absolute;
        top: 36px;
        right: 10px;
        background: rgba(255,255,255,.92);
        border: 1px solid rgba(0,0,0,.12);
        border-radius: 12px;
        box-shadow: 0 14px 30px rgba(0,0,0,.2);
        padding: 8px;
        display: none;
        gap: 8px;
        flex-wrap: wrap;
        width: 190px;
        z-index: ${Z};
      }
      .${UI_NS}-palette.${UI_NS}-show{ display: flex; }
      .${UI_NS}-swatch{
        width: 26px;
        height: 26px;
        border-radius: 10px;
        border: 1px solid rgba(0,0,0,.15);
        cursor: pointer;
      }
      .${UI_NS}-resize{
        position: absolute;
        right: 2px;
        bottom: 2px;
        width: 18px;
        height: 18px;
        cursor: nwse-resize;
        opacity: .6;
      }
      .${UI_NS}-resize:before{
        content:'';
        position:absolute;
        inset: 5px 3px 3px 5px;
        border-right: 2px solid rgba(0,0,0,.35);
        border-bottom: 2px solid rgba(0,0,0,.35);
        border-radius: 2px;
      }
      .${UI_NS}-minimized .${UI_NS}-body{ display:none; }
      .${UI_NS}-minimized{ height: 34px !important; }
      .${UI_NS}-sr{ position:absolute; left:-9999px; top:auto; width:1px; height:1px; overflow:hidden; }
    `;
    document.documentElement.appendChild(style);
  }

  async function storageGetAll() {
    return await new Promise((resolve) => {
      chrome.storage.local.get([STORAGE_KEY], (res) => {
        resolve(res?.[STORAGE_KEY] || {});
      });
    });
  }

  async function storageSetAll(all) {
    return await new Promise((resolve) => {
      chrome.storage.local.set({ [STORAGE_KEY]: all }, () => resolve());
    });
  }

  async function getNotesForHost(host) {
    const all = await storageGetAll();
    const arr = Array.isArray(all?.[host]) ? all[host] : [];
    return arr;
  }

  async function setNotesForHost(host, arr) {
    const all = await storageGetAll();
    all[host] = arr;
    await storageSetAll(all);
  }

  function scheduleSaveNote(host, note) {
    // debounce saves per-note to keep typing smooth
    const id = note.id;
    if (state.saveTimers.has(id)) clearTimeout(state.saveTimers.get(id));
    state.saveTimers.set(id, setTimeout(async () => {
      state.saveTimers.delete(id);
      await persistNote(host, note);
    }, 220));
  }

  async function persistNote(host, note) {
    const arr = await getNotesForHost(host);
    const idx = arr.findIndex(n => n.id === note.id);
    const clean = sanitizeNote(note);
    if (idx >= 0) arr[idx] = clean;
    else arr.push(clean);
    await setNotesForHost(host, arr);
  }

  async function removeNote(host, id) {
    const arr = await getNotesForHost(host);
    const next = arr.filter(n => n.id !== id);
    await setNotesForHost(host, next);
  }

  function sanitizeNote(note) {
    const n = { ...DEFAULT_NOTE, ...note };
    n.id = String(note.id || uid());
    n.text = String(n.text || '').slice(0, 12000);
    n.x = Number.isFinite(n.x) ? n.x : DEFAULT_NOTE.x;
    n.y = Number.isFinite(n.y) ? n.y : DEFAULT_NOTE.y;
    n.w = Number.isFinite(n.w) ? n.w : DEFAULT_NOTE.w;
    n.h = Number.isFinite(n.h) ? n.h : DEFAULT_NOTE.h;
    n.w = clamp(n.w, 180, 900);
    n.h = clamp(n.h, 34, 900);
    n.color = String(n.color || DEFAULT_NOTE.color);
    n.minimized = !!n.minimized;
    n.createdAt = Number.isFinite(n.createdAt) ? n.createdAt : now();
    n.updatedAt = Number.isFinite(n.updatedAt) ? n.updatedAt : now();
    n.origin = n.origin || state.origin || '';
    return n;
  }

  function applyNoteStyles(root, note) {
    root.style.setProperty('--x', `${note.x}px`);
    root.style.setProperty('--y', `${note.y}px`);
    root.style.transform = `translate3d(${note.x}px, ${note.y}px, 0)`;
    root.style.width = `${note.w}px`;
    root.style.height = `${note.h}px`;
    root.style.background = note.color;
    const isDark = note.color.toLowerCase() === '#1e1e1e' || note.color.toLowerCase() === '#000' || note.color.toLowerCase() === 'black';
    root.classList.toggle(`${UI_NS}-dark`, isDark);
    root.classList.toggle(`${UI_NS}-minimized`, !!note.minimized);
  }

  function closePalettesExcept(exceptId) {
    for (const [id, els] of state.els.entries()) {
      if (id !== exceptId) els.palette.classList.remove(`${UI_NS}-show`);
    }
  }

  function createNoteElement(note) {
    ensureStyles();

    const root = document.createElement('div');
    root.className = `${UI_NS}-note`;
    root.dataset.noteId = note.id;

    const header = document.createElement('div');
    header.className = `${UI_NS}-header`;

    const title = document.createElement('div');
    title.className = `${UI_NS}-title`;
    title.innerHTML = `<span>פתק</span>`;
    const badge = document.createElement('span');
    badge.className = `${UI_NS}-badge`;
    badge.textContent = state.host || 'אתר';
    title.appendChild(badge);

    const btns = document.createElement('div');
    btns.className = `${UI_NS}-btns`;

    const btnColor = document.createElement('button');
    btnColor.type = 'button';
    btnColor.className = `${UI_NS}-iconbtn`;
    btnColor.title = 'שנה צבע';
    btnColor.textContent = '🎨';

    const btnMin = document.createElement('button');
    btnMin.type = 'button';
    btnMin.className = `${UI_NS}-iconbtn`;
    btnMin.title = 'מזער / הגדל';
    btnMin.textContent = '—';

    const btnClose = document.createElement('button');
    btnClose.type = 'button';
    btnClose.className = `${UI_NS}-iconbtn`;
    btnClose.title = 'סגור (לא מוחק)';
    btnClose.textContent = '✕';

    const btnDel = document.createElement('button');
    btnDel.type = 'button';
    btnDel.className = `${UI_NS}-iconbtn`;
    btnDel.title = 'מחק פתק';
    btnDel.textContent = '🗑';

    btns.appendChild(btnColor);
    btns.appendChild(btnMin);
    btns.appendChild(btnClose);
    btns.appendChild(btnDel);

    header.appendChild(title);
    header.appendChild(btns);

    const body = document.createElement('div');
    body.className = `${UI_NS}-body`;

    const textarea = document.createElement('textarea');
    textarea.className = `${UI_NS}-text`;
    textarea.placeholder = 'כתוב כאן…';
    textarea.value = note.text || '';
    textarea.setAttribute('aria-label', 'תוכן הפתק');

    body.appendChild(textarea);

    const palette = document.createElement('div');
    palette.className = `${UI_NS}-palette`;
    for (const c of COLORS) {
      const sw = document.createElement('div');
      sw.className = `${UI_NS}-swatch`;
      sw.style.background = c;
      sw.title = c === '#1E1E1E' ? 'כהה' : 'בחר צבע';
      sw.addEventListener('click', () => {
        note.color = c;
        note.updatedAt = now();
        applyNoteStyles(root, note);
        scheduleSaveNote(state.host, note);
        palette.classList.remove(`${UI_NS}-show`);
      });
      palette.appendChild(sw);
    }

    const resize = document.createElement('div');
    resize.className = `${UI_NS}-resize`;
    resize.title = 'שנה גודל';

    root.appendChild(header);
    root.appendChild(body);
    root.appendChild(palette);
    root.appendChild(resize);

    applyNoteStyles(root, note);

    // Drag
    let dragging = false;
    let dragStartX = 0, dragStartY = 0;
    let noteStartX = 0, noteStartY = 0;

    header.addEventListener('pointerdown', (e) => {
      // ignore clicks on buttons
      const t = e.target;
      if (t && t.closest && t.closest(`.${UI_NS}-iconbtn`)) return;
      dragging = true;
      header.setPointerCapture(e.pointerId);
      header.style.cursor = 'grabbing';
      dragStartX = e.clientX;
      dragStartY = e.clientY;
      noteStartX = note.x;
      noteStartY = note.y;
      closePalettesExcept(note.id);
      e.preventDefault();
    });

    header.addEventListener('pointermove', (e) => {
      if (!dragging) return;
      const dx = e.clientX - dragStartX;
      const dy = e.clientY - dragStartY;
      note.x = clamp(noteStartX + dx, 0, Math.max(0, window.innerWidth - 120));
      note.y = clamp(noteStartY + dy, 0, Math.max(0, window.innerHeight - 44));
      applyNoteStyles(root, note);
    });

    header.addEventListener('pointerup', (e) => {
      if (!dragging) return;
      dragging = false;
      header.style.cursor = 'grab';
      note.updatedAt = now();
      scheduleSaveNote(state.host, note);
    });

    // Resize
    let resizing = false;
    let rsx = 0, rsy = 0, rW = 0, rH = 0;

    resize.addEventListener('pointerdown', (e) => {
      resizing = true;
      resize.setPointerCapture(e.pointerId);
      rsx = e.clientX; rsy = e.clientY;
      rW = root.getBoundingClientRect().width;
      rH = root.getBoundingClientRect().height;
      closePalettesExcept(note.id);
      e.preventDefault();
    });

    resize.addEventListener('pointermove', (e) => {
      if (!resizing) return;
      const dw = e.clientX - rsx;
      const dh = e.clientY - rsy;
      const newW = clamp(rW + dw, 180, Math.min(900, window.innerWidth - 8));
      const newH = clamp(rH + dh, 34, Math.min(900, window.innerHeight - 8));
      note.w = newW;
      note.h = newH;
      applyNoteStyles(root, note);
    });

    resize.addEventListener('pointerup', () => {
      if (!resizing) return;
      resizing = false;
      note.updatedAt = now();
      scheduleSaveNote(state.host, note);
    });

    // Text autosave
    textarea.addEventListener('input', () => {
      note.text = textarea.value;
      note.updatedAt = now();
      scheduleSaveNote(state.host, note);
    });

    // Buttons
    btnColor.addEventListener('click', () => {
      const show = !palette.classList.contains(`${UI_NS}-show`);
      closePalettesExcept(note.id);
      palette.classList.toggle(`${UI_NS}-show`, show);
    });

    btnMin.addEventListener('click', () => {
      note.minimized = !note.minimized;
      // if minimizing, store previous height so expanding returns
      if (note.minimized) {
        note._prevH = note.h;
        note.h = 34;
      } else {
        if (Number.isFinite(note._prevH) && note._prevH >= 120) note.h = note._prevH;
        delete note._prevH;
      }
      note.updatedAt = now();
      applyNoteStyles(root, note);
      scheduleSaveNote(state.host, note);
    });

    btnClose.addEventListener('click', () => {
      // close for this tab only (not delete)
      root.remove();
      state.els.delete(note.id);
    });

    btnDel.addEventListener('click', async () => {
      await removeNote(state.host, note.id);
      root.remove();
      state.notes.delete(note.id);
      state.els.delete(note.id);
    });

    // Click outside to close palettes
    root.addEventListener('click', (e) => {
      const t = e.target;
      if (!t.closest(`.${UI_NS}-iconbtn`) && !t.closest(`.${UI_NS}-palette`)) {
        palette.classList.remove(`${UI_NS}-show`);
      }
    });

    // Keep in viewport on resize
    window.addEventListener('resize', () => {
      const maxX = Math.max(0, window.innerWidth - 120);
      const maxY = Math.max(0, window.innerHeight - 44);
      note.x = clamp(note.x, 0, maxX);
      note.y = clamp(note.y, 0, maxY);
      applyNoteStyles(root, note);
      scheduleSaveNote(state.host, note);
    }, { passive: true });

    return { root, textarea, palette };
  }

  async function renderExistingNotes() {
    if (!state.host) return;
    const arr = await getNotesForHost(state.host);
    for (const raw of arr) {
      const note = sanitizeNote(raw);
      state.notes.set(note.id, note);
      const els = createNoteElement(note);
      state.els.set(note.id, els);
      document.documentElement.appendChild(els.root);
    }
  }

  async function createNewNote(focus = true) {
    if (!state.host) return;
    const arr = await getNotesForHost(state.host);
    const note = sanitizeNote({
      ...DEFAULT_NOTE,
      id: uid(),
      origin: state.origin,
      createdAt: now(),
      updatedAt: now(),
      // offset new notes so they don't stack perfectly
      x: clamp(24 + (arr.length * 18), 0, Math.max(0, window.innerWidth - 120)),
      y: clamp(24 + (arr.length * 18), 0, Math.max(0, window.innerHeight - 44))
    });

    arr.push(note);
    await setNotesForHost(state.host, arr);

    state.notes.set(note.id, note);
    const els = createNoteElement(note);
    state.els.set(note.id, els);
    document.documentElement.appendChild(els.root);
    if (focus) els.textarea.focus();
  }

  async function deleteNoteById(id) {
    if (!state.host) return;
    await removeNote(state.host, id);
    const els = state.els.get(id);
    if (els?.root) els.root.remove();
    state.els.delete(id);
    state.notes.delete(id);
  }

  async function focusNoteById(id) {
    const els = state.els.get(id);
    if (!els) return;
    els.root.style.boxShadow = '0 20px 50px rgba(0,0,0,.35)';
    els.root.scrollIntoView({ block: 'center', inline: 'center' });
    setTimeout(() => {
      if (els.root) els.root.style.boxShadow = '';
    }, 900);
    els.textarea.focus();
  }

  // Messages from popup
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    (async () => {
      if (!msg || typeof msg !== 'object') return;

      if (msg.type === 'SSN_CREATE_NOTE') {
        await createNewNote(true);
        sendResponse({ ok: true });
      } else if (msg.type === 'SSN_DELETE_NOTE' && msg.id) {
        await deleteNoteById(String(msg.id));
        sendResponse({ ok: true });
      } else if (msg.type === 'SSN_FOCUS_NOTE' && msg.id) {
        await focusNoteById(String(msg.id));
        sendResponse({ ok: true });
      } else if (msg.type === 'SSN_PING') {
        sendResponse({ ok: true, host: state.host });
      }
    })();
    return true; // async
  });

  // Init
  renderExistingNotes().catch(() => {});
})();
