# -*- coding: utf-8 -*-
"""
=== SMS AI Bot Server (Final Version with Key Rotation) ===

הוראות התקנה (עבור החברים):
1. וודאו שמותקן Python במחשב.
2. פתחו את הטרמינל/CMD והריצו את הפקודה הבאה להתקנת הספריות:
   pip install flask requests

3. הריצו את הקובץ:
   python app.py

4. כנסו בדפדפן לכתובת: http://127.0.0.1:5000
5. בצעו את ההגדרה הראשונית. בשדה ה-API ניתן להכניס מספר מפתחות מופרדים בפסיק.
"""

import os
import csv
import json
import time
import threading
import logging
import requests
import re
import traceback
from datetime import datetime
from flask import Flask, jsonify, request, render_template_string, redirect, url_for

# --- הגדרות מערכת ---
# הערה: הגדרות משתמש (סיסמאות וכו') נטענות כעת מקובץ config.json

# --- קבועים ---
YEMOT_API_URL = "https://www.call2all.co.il/ym/api"
CURRENCY_API_URL = "https://api.frankfurter.app/latest?from=USD&to=ILS"
WEATHER_API_URL = "https://api.open-meteo.com/v1/forecast"
ZMANIM_API_URL_BASE = "https://www.hebcal.com/zmanim?cfg=json&tzid=Asia/Jerusalem"

DEFAULT_HISTORY_LIMIT = 20 # כמות הודעות אחרונות לזיכרון כברירת מחדל
MAX_RESPONSE_LENGTH = 190
POLLING_INTERVAL_SECONDS = 5

# הגדרות ברירת מחדל - ניסוח מחייב יותר כהנחיית מערכת
DEFAULT_PROMPT_SUFFIX = "אתה עוזר בוט ב-SMS. עליך לענות תשובות קצרות ותמציתיות בלבד! אורך התשובה המקסימלי הוא 70 תווים. במקרים חריגים ניתן להגיע ל-130 תווים. אל תכתוב הקדמות, אל תציג את עצמך, וגש ישר לתשובה."
DEFAULT_MODEL_NAME = "2.5 flash"
DEFAULT_WEATHER_CITY = "תל אביב"
DEFAULT_ZMANIM_CITY = "תל אביב"

# מילון הגדרות זמנים
ALL_ZMANIM_OPTIONS = [
    {'id': 'alot', 'hebcal_key': 'alotHaShachar', 'label': 'עלות השחר'},
    {'id': 'misheyakir', 'hebcal_key': 'misheyakir', 'label': 'משיכיר (טלית ותפילין)'},
    {'id': 'netz', 'hebcal_key': 'sunrise', 'label': 'נץ החמה'},
    {'id': 'shma_mga', 'hebcal_key': 'sofZmanShmaMGA', 'label': 'סזק"ש (מג"א)'},
    {'id': 'shma_gra', 'hebcal_key': 'sofZmanShma', 'label': 'סזק"ש (גר"א/אדה"ז)'},
    {'id': 'tefila_mga', 'hebcal_key': 'sofZmanTfilaMGA', 'label': 'סוף זמן תפילה (מג"א)'},
    {'id': 'tefila_gra', 'hebcal_key': 'sofZmanTfila', 'label': 'סוף זמן תפילה (גר"א/אדה"ז)'},
    {'id': 'chatzot', 'hebcal_key': 'chatzot', 'label': 'חצות היום'},
    {'id': 'mincha_gedola', 'hebcal_key': 'minchaGedola', 'label': 'מנחה גדולה'},
    {'id': 'mincha_ketana', 'hebcal_key': 'minchaKetana', 'label': 'מנחה קטנה'},
    {'id': 'plag', 'hebcal_key': 'plagHaMincha', 'label': 'פלג המנחה'},
    {'id': 'shkia', 'hebcal_key': 'sunset', 'label': 'שקיעה'},
    {'id': 'tzeit', 'hebcal_key': 'tzeit7083deg', 'label': 'צאת הכוכבים'}
]

DEFAULT_ENABLED_ZMANIM = ['netz', 'shma_mga', 'shma_gra', 'tefila_gra', 'chatzot', 'shkia', 'tzeit']

# --- מילונים ---
GEMINI_MODELS = {
    "2.5 pro": "gemini-2.5-pro",
    "2.5 flash": "gemini-2.5-flash",
    "2.5 flash lite": "gemini-2.5-flash-lite-preview-06-17",
    "2.0 flash": "gemini-2.0-flash-001"
}

WEATHER_CITIES = {
    "תל אביב": {"lat": 32.08, "lon": 34.78},
    "ירושלים": {"lat": 31.77, "lon": 35.22},
    "חיפה": {"lat": 32.82, "lon": 34.99},
    "באר שבע": {"lat": 31.25, "lon": 34.79},
    "קריית גת": {"lat": 31.61, "lon": 34.77},
    "אילת": {"lat": 29.56, "lon": 34.95}
}

ZMANIM_CITIES = {
    "תל אביב": "293397",
    "ירושלים": "281184",
    "באר שבע": "295530",
    "קריית גת": "294131"
}

# --- נתיבי קבצים ---
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
HISTORY_FILE = os.path.join(SCRIPT_DIR, "conversation_history.json")
CREDITS_FILE = os.path.join(SCRIPT_DIR, "credits.csv")
CONFIG_FILE = os.path.join(SCRIPT_DIR, "config.json")

# --- משתנים גלובליים לזמן ריצה ---
server_logs = []
PROCESSED_SMS_IDS = set()
USER_STATES = {}
APP_START_TIME = datetime.now()
bot_active = True

# ==============================================================================
#                              ממשק HTML (React)
# ==============================================================================
HTML_TEMPLATE = """
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>מערכת ניהול SMS AI</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/react@18/umd/react.production.min.js"></script>
    <script src="https://unpkg.com/react-dom@18/umd/react-dom.production.min.js"></script>
    <script src="https://unpkg.com/babel-standalone@6/babel.min.js"></script>
    <script src="https://unpkg.com/lucide@latest"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Heebo:wght@300;400;600;700&display=swap');
        body { font-family: 'Heebo', sans-serif; background-color: #f3f4f6; }
        select, input, textarea { background-color: #f8fafc; border: 1px solid #e2e8f0; border-radius: 0.375rem; }
        .code-block { direction: ltr; font-family: monospace; background: #1e293b; color: #e2e8f0; padding: 1rem; border-radius: 0.5rem; overflow-x: auto; }
        
        /* עיצוב לוגים צבעוני */
        .log-error { color: #ef4444; background-color: #fef2f2; }
        .log-success { color: #15803d; background-color: #f0fdf4; }
        .log-info { color: #334155; }
    </style>
</head>
<body>
    <div id="root"></div>

    <script type="text/babel">
        const { useState, useEffect } = React;
        
        const MODELS = ["2.5 pro", "2.5 flash", "2.5 flash lite", "2.0 flash"];
        const CITIES = ["תל אביב", "ירושלים", "חיפה", "באר שבע", "קריית גת", "אילת"];
        
        const ZMANIM_OPTIONS = [
            {id: 'alot', label: 'עלות השחר'},
            {id: 'misheyakir', label: 'משיכיר (טלית ותפילין)'},
            {id: 'netz', label: 'נץ החמה'},
            {id: 'shma_mga', label: 'סזק"ש (מג"א)'},
            {id: 'shma_gra', label: 'סזק"ש (גר"א/אדה"ז)'},
            {id: 'tefila_mga', label: 'סוף זמן תפילה (מג"א)'},
            {id: 'tefila_gra', label: 'סוף זמן תפילה (גר"א/אדה"ז)'},
            {id: 'chatzot', label: 'חצות היום'},
            {id: 'mincha_gedola', label: 'מנחה גדולה'},
            {id: 'mincha_ketana', label: 'מנחה קטנה'},
            {id: 'plag', label: 'פלג המנחה'},
            {id: 'shkia', label: 'שקיעה'},
            {id: 'tzeit', label: 'צאת הכוכבים'}
        ];

        const Icons = {
            Users: () => <i data-lucide="users" className="w-6 h-6"></i>,
            CreditCard: () => <i data-lucide="credit-card" className="w-6 h-6"></i>,
            Activity: () => <i data-lucide="activity" className="w-6 h-6"></i>,
            Settings: () => <i data-lucide="settings" className="w-6 h-6"></i>,
            Trash: () => <i data-lucide="trash-2" className="w-4 h-4"></i>,
            Refresh: () => <i data-lucide="refresh-cw" className="w-4 h-4"></i>,
            LogOut: () => <i data-lucide="log-out" className="w-4 h-4"></i>,
            BookOpen: () => <i data-lucide="book-open" className="w-6 h-6"></i>,
            FileText: () => <i data-lucide="file-text" className="w-6 h-6"></i>,
            Copy: () => <i data-lucide="copy" className="w-4 h-4"></i>,
            Clock: () => <i data-lucide="clock" className="w-6 h-6"></i>,
            List: () => <i data-lucide="list" className="w-6 h-6"></i>
        };

        function SetupScreen() {
            const [formData, setFormData] = useState({
                yemot_username: '',
                yemot_password: '',
                gemini_api_key: '',
                admin_phone: ''
            });

            const handleSubmit = async (e) => {
                e.preventDefault();
                await fetch('/api/setup', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(formData)
                });
                window.location.reload();
            };

            return (
                <div className="min-h-screen flex items-center justify-center bg-slate-900 p-4">
                    <div className="bg-white p-8 rounded-xl shadow-2xl max-w-md w-full">
                        <h1 className="text-2xl font-bold mb-6 text-center text-slate-800">הגדרה ראשונית</h1>
                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div><label className="block text-sm font-medium mb-1">מספר מערכת (ימות המשיח)</label><input required className="w-full p-2" value={formData.yemot_username} onChange={e => setFormData({...formData, yemot_username: e.target.value})} /></div>
                            <div><label className="block text-sm font-medium mb-1">סיסמה (ימות המשיח)</label><input required type="password" className="w-full p-2" value={formData.yemot_password} onChange={e => setFormData({...formData, yemot_password: e.target.value})} /></div>
                            <div>
                                <label className="block text-sm font-medium mb-1">מפתח Gemini API (ניתן להזין כמה מופרדים בפסיק)</label>
                                <input required type="password" className="w-full p-2" value={formData.gemini_api_key} onChange={e => setFormData({...formData, gemini_api_key: e.target.value})} placeholder="AIzaSy..., AIzaSy..." />
                            </div>
                            <div><label className="block text-sm font-medium mb-1">טלפון מנהל (ללא מקף, פורמט 972)</label><input required className="w-full p-2" placeholder="972501234567" value={formData.admin_phone} onChange={e => setFormData({...formData, admin_phone: e.target.value})} /></div>
                            <button className="w-full bg-blue-600 text-white p-3 rounded font-bold hover:bg-blue-700 transition">שמור והתחל</button>
                        </form>
                    </div>
                </div>
            );
        }

        function GuideTab() {
            return (
                <div className="bg-white rounded-xl shadow-sm border border-slate-200 p-8 space-y-6">
                    <div>
                        <h2 className="text-2xl font-bold text-slate-800 mb-2">מדריך למשתמש</h2>
                        <p className="text-slate-600">הסבר על אופן פעולת המערכת והפקודות הזמינות ב-SMS.</p>
                    </div>

                    <div className="grid md:grid-cols-2 gap-6">
                        <div className="bg-blue-50 p-5 rounded-lg border border-blue-100">
                            <h3 className="font-bold text-blue-800 mb-3 flex items-center gap-2"><Icons.Activity/> פקודות כלליות</h3>
                            <ul className="space-y-2 text-sm text-blue-900">
                                <li><strong>שאלה חופשית:</strong> פשוט שלח כל שאלה וקבל תשובה מה-AI.</li>
                                <li><strong>דולר / שער הדולר:</strong> קבלת השער היציג העדכני.</li>
                                <li><strong>100 דולר:</strong> המרת סכום ספציפי לשקלים.</li>
                                <li><strong>מזג אוויר:</strong> תחזית ל-3 ימים לעיר המוגדרת שלך.</li>
                                <li><strong>זמני היום / זמנים:</strong> זמני כניסה/יציאה לעיר שלך.</li>
                                <li><strong>סטטוס:</strong> בדיקת יתרת קרדיט והמודל הפעיל.</li>
                                <li><strong>איפוס:</strong> מחיקת היסטוריית השיחה (זיכרון) עם הבוט.</li>
                            </ul>
                        </div>
                        
                        <div className="bg-purple-50 p-5 rounded-lg border border-purple-100">
                            <h3 className="font-bold text-purple-800 mb-3 flex items-center gap-2"><Icons.Settings/> הגדרות אישיות (דרך SMS)</h3>
                            <ul className="space-y-2 text-sm text-purple-900">
                                <li><strong>שינוי מודל:</strong> פותח תפריט לבחירת מודל (Pro, Flash וכו').</li>
                                <li><strong>שינוי אזור:</strong> בחירת עיר לתחזית מזג אוויר.</li>
                                <li><strong>שינוי אזור זמנים:</strong> בחירת עיר לזמני היום.</li>
                            </ul>
                        </div>
                    </div>
                     <div className="text-sm text-slate-500 mt-4 border-t pt-4">
                        <strong>טיפ למנהלים:</strong> ניתן להגדיר אלו זמנים יוצגו בפקודת "זמנים" דרך לשונית ההגדרות באתר.
                    </div>
                </div>
            );
        }

        function SettingsTab({ config, onSaveConfig }) {
            const [promptSuffix, setPromptSuffix] = useState('');
            const [defaultPrompt, setDefaultPrompt] = useState('');
            const [selectedZmanim, setSelectedZmanim] = useState([]);
            const [historyLimit, setHistoryLimit] = useState(20);

            useEffect(() => {
                if(config) {
                    setPromptSuffix(config.prompt_suffix || '');
                    setDefaultPrompt(config.default_prompt_suffix || 'לא זמין');
                    setSelectedZmanim(config.zmanim_settings || ['netz', 'shma_mga', 'shma_gra', 'tefila_gra', 'chatzot', 'shkia', 'tzeit']);
                    setHistoryLimit(config.history_limit || 20);
                }
            }, [config]);

            const handleSave = () => {
                const newConfig = { 
                    ...config, 
                    prompt_suffix: promptSuffix,
                    zmanim_settings: selectedZmanim,
                    history_limit: parseInt(historyLimit)
                };
                onSaveConfig(newConfig);
                alert('ההגדרות נשמרו בהצלחה!');
            };

            const toggleZman = (id) => {
                if (selectedZmanim.includes(id)) {
                    setSelectedZmanim(selectedZmanim.filter(z => z !== id));
                } else {
                    setSelectedZmanim([...selectedZmanim, id]);
                }
            };

            const copyDefault = () => setPromptSuffix(defaultPrompt);

            return (
                <div className="space-y-6">
                    <div className="bg-white rounded-xl shadow-sm border border-slate-200 p-8">
                        <h2 className="text-xl font-bold text-slate-800 mb-6 flex items-center gap-2"><Icons.List/> הגדרות זיכרון והיסטוריה</h2>
                        <div>
                            <label className="block text-sm font-bold text-slate-700 mb-2">אורך זיכרון שיחה (מספר הודעות אחרונות)</label>
                            <input 
                                type="number" 
                                min="2" max="200"
                                className="w-24 p-2 border rounded"
                                value={historyLimit}
                                onChange={(e) => setHistoryLimit(e.target.value)}
                            />
                            <p className="text-xs text-slate-500 mt-1">קובע כמה הודעות אחרונות הבוט "זוכר" בכל שיחה. מספר גבוה יותר זוכר יותר הקשר אך עולה יותר בטוקנים.</p>
                        </div>
                    </div>

                    <div className="bg-white rounded-xl shadow-sm border border-slate-200 p-8">
                        <h2 className="text-xl font-bold text-slate-800 mb-6 flex items-center gap-2"><Icons.Clock/> הגדרת זמני היום להצגה</h2>
                        <p className="text-xs text-slate-500 mb-4">סמן את הזמנים שברצונך שהמערכת תשלח למשתמשים בפקודת "זמנים". הסדר כאן הוא הסדר שיוצג.</p>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                            {ZMANIM_OPTIONS.map(opt => (
                                <label key={opt.id} className="flex items-center gap-2 p-2 border rounded hover:bg-slate-50 cursor-pointer">
                                    <input 
                                        type="checkbox" 
                                        checked={selectedZmanim.includes(opt.id)} 
                                        onChange={() => toggleZman(opt.id)}
                                        className="w-4 h-4 text-blue-600 rounded"
                                    />
                                    <span className="text-sm text-slate-700">{opt.label}</span>
                                </label>
                            ))}
                        </div>
                    </div>

                    <div className="bg-white rounded-xl shadow-sm border border-slate-200 p-8">
                        <h2 className="text-xl font-bold text-slate-800 mb-6 flex items-center gap-2"><Icons.Settings/> הגדרות מערכת מתקדמות</h2>
                        
                        <div className="mb-6">
                            <label className="block text-sm font-bold text-slate-700 mb-2">הנחיית מערכת פעילה (System Prompt)</label>
                            <p className="text-xs text-slate-500 mb-2">
                                טקסט זה נשלח ל-Gemini בכל בקשה כהנחיית על.
                            </p>
                            <textarea 
                                className="w-full p-3 h-24 text-sm font-mono border-slate-300 focus:border-blue-500 focus:ring-1 focus:ring-blue-500 mb-2"
                                value={promptSuffix}
                                onChange={(e) => setPromptSuffix(e.target.value)}
                                placeholder={defaultPrompt}
                            />
                        </div>

                        <div className="mb-6">
                             <div className="flex justify-between items-end mb-2">
                                 <label className="block text-sm font-bold text-slate-700">ערך ברירת המחדל (לעיון והעתקה)</label>
                                 <button onClick={copyDefault} className="text-xs text-blue-600 hover:text-blue-800 flex items-center gap-1"><Icons.Copy className="w-3 h-3"/> העתק לתיבה למעלה</button>
                             </div>
                             <div className="code-block text-xs">
                                {defaultPrompt}
                             </div>
                        </div>

                        <button onClick={handleSave} className="bg-blue-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-blue-700 transition flex items-center gap-2">
                            <Icons.Settings className="w-4 h-4" /> שמור כל השינויים
                        </button>
                    </div>
                </div>
            );
        }

        function App() {
            const [isConfigured, setIsConfigured] = useState(null);
            const [activeTab, setActiveTab] = useState('dashboard');
            const [users, setUsers] = useState([]);
            const [logs, setLogs] = useState([]);
            const [stats, setStats] = useState({ total_users: 0, total_requests: 0, balance: 'טוען...' });
            const [config, setConfig] = useState({});

            const checkConfig = async () => {
                const res = await fetch('/api/status');
                const data = await res.json();
                setIsConfigured(data.configured);
                if(data.configured) fetchData();
            };

            const fetchData = async () => {
                try {
                    const [resUsers, resStats, resLogs] = await Promise.all([
                        fetch('/api/users'), fetch('/api/stats'), fetch('/api/logs')
                    ]);
                    setUsers(await resUsers.json());
                    setStats(await resStats.json());
                    setLogs(await resLogs.json());
                } catch (e) { console.error("Error fetching data", e); }
            };

            useEffect(() => {
                checkConfig();
                // טעינת הגדרות פעם אחת בלבד בעת הטעינה
                fetch('/api/config').then(r=>r.json()).then(setConfig);

                const interval = setInterval(() => {
                    fetch('/api/status').then(r=>r.json()).then(d => {
                        // רענון רק של נתונים דינמיים ולא הגדרות
                        if(d.configured) fetchData();
                    });
                }, 3000);
                return () => clearInterval(interval);
            }, []);

            useEffect(() => { lucide.createIcons(); }, [users, activeTab, isConfigured]);

            const updateUserField = async (phone, field, value) => {
                await fetch('/api/update_user', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ phone, field, value })
                });
                fetchData();
            };

            const deleteUser = async (phone) => {
                if(!confirm('למחוק משתמש זה?')) return;
                await fetch('/api/delete_user', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ phone })
                });
                fetchData();
            };

            const saveConfig = async (newConfig) => {
                await fetch('/api/update_config', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(newConfig)
                });
                // עדכון לוקאלי מיידי כדי למנוע הבהוב
                setConfig(newConfig);
            };

            const resetConfig = async () => {
                if(!confirm('האם אתה בטוח? תצטרך להזין סיסמאות מחדש.')) return;
                await fetch('/api/reset_config', { method: 'POST' });
                window.location.reload();
            };
            
            const getLogClass = (msg) => {
                if (msg.includes('שגיאה') || msg.includes('Error') || msg.includes('נכשל')) return 'log-error';
                if (msg.includes('נשלח') || msg.includes('עודכן')) return 'log-success';
                return 'log-info';
            };

            if (isConfigured === false) return <SetupScreen />;
            if (isConfigured === null) return <div className="flex items-center justify-center h-screen text-slate-500">טוען מערכת...</div>;

            return (
                <div className="min-h-screen flex flex-col md:flex-row">
                    <div className="w-full md:w-64 bg-slate-900 text-white p-6 flex flex-col">
                        <h1 className="text-2xl font-bold mb-8 text-blue-400">🤖 SMS AI Bot</h1>
                        <nav className="flex-1 space-y-2">
                            <button onClick={() => setActiveTab('dashboard')} className={`w-full flex items-center gap-3 p-3 rounded-lg transition ${activeTab === 'dashboard' ? 'bg-blue-600' : 'hover:bg-slate-800'}`}>
                                <Icons.Activity /> דשבורד
                            </button>
                            <button onClick={() => setActiveTab('users')} className={`w-full flex items-center gap-3 p-3 rounded-lg transition ${activeTab === 'users' ? 'bg-blue-600' : 'hover:bg-slate-800'}`}>
                                <Icons.Users /> משתמשים
                            </button>
                            <button onClick={() => setActiveTab('guide')} className={`w-full flex items-center gap-3 p-3 rounded-lg transition ${activeTab === 'guide' ? 'bg-blue-600' : 'hover:bg-slate-800'}`}>
                                <Icons.BookOpen /> מדריך
                            </button>
                            <button onClick={() => setActiveTab('settings')} className={`w-full flex items-center gap-3 p-3 rounded-lg transition ${activeTab === 'settings' ? 'bg-blue-600' : 'hover:bg-slate-800'}`}>
                                <Icons.Settings /> הגדרות
                            </button>
                        </nav>
                         <div className="mt-auto pt-4 border-t border-slate-700">
                            <button onClick={resetConfig} className="w-full flex items-center gap-2 text-sm text-red-400 hover:text-red-300 transition">
                                <Icons.LogOut /> יציאה / איפוס
                            </button>
                        </div>
                    </div>

                    <div className="flex-1 p-6 md:p-10 overflow-y-auto">
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                            <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200">
                                <div className="flex justify-between items-center mb-4"><h3 className="text-slate-500 font-medium">משתמשים רשומים</h3><span className="p-2 bg-blue-100 text-blue-600 rounded-lg"><Icons.Users /></span></div>
                                <div className="text-3xl font-bold text-slate-800">{users.length}</div>
                            </div>
                            <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200">
                                <div className="flex justify-between items-center mb-4"><h3 className="text-slate-500 font-medium">הודעות שטופלו</h3><span className="p-2 bg-green-100 text-green-600 rounded-lg"><Icons.Activity /></span></div>
                                <div className="text-3xl font-bold text-slate-800">{stats.total_requests}</div>
                            </div>
                            <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-200">
                                <div className="flex justify-between items-center mb-4"><h3 className="text-slate-500 font-medium">סטטוס בוט</h3><span className="p-2 bg-purple-100 text-purple-600 rounded-lg"><Icons.CreditCard /></span></div>
                                <div className="text-xl font-bold text-slate-800">{stats.balance}</div>
                            </div>
                        </div>

                        {activeTab === 'dashboard' && (
                            <div className="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden">
                                <div className="p-6 border-b border-slate-100 bg-slate-50 flex justify-between items-center"><h2 className="text-lg font-bold text-slate-800">לוג פעילות אחרון</h2></div>
                                <div>
                                    {logs.length === 0 ? <p className="p-6 text-slate-400">ממתין לפעילות...</p> : (
                                        <ul className="divide-y divide-slate-100">
                                            {logs.slice().reverse().map((log, i) => (
                                                <li key={i} className={`p-4 hover:bg-slate-50 transition flex gap-4 ${getLogClass(log.msg)}`}>
                                                    <span className="text-xs font-mono text-slate-500 bg-slate-100 px-2 py-1 rounded h-fit">{log.time}</span>
                                                    <span className="text-sm font-medium">{log.msg}</span>
                                                </li>
                                            ))}
                                        </ul>
                                    )}
                                </div>
                            </div>
                        )}

                        {activeTab === 'users' && (
                            <div className="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden">
                                <div className="p-6 border-b border-slate-100 bg-slate-50"><h2 className="text-lg font-bold text-slate-800">ניהול משתמשים והגדרות</h2></div>
                                <div className="overflow-x-auto">
                                    <table className="w-full text-sm text-right">
                                        <thead className="bg-slate-50 text-slate-500 font-medium">
                                            <tr>
                                                <th className="p-4">טלפון</th>
                                                <th className="p-4">מודל</th>
                                                <th className="p-4">קרדיט</th>
                                                <th className="p-4">עיר מזג אוויר</th>
                                                <th className="p-4">עיר זמנים</th>
                                                <th className="p-4">פעולות</th>
                                            </tr>
                                        </thead>
                                        <tbody className="divide-y divide-slate-100">
                                            {users.map((user, idx) => (
                                                <tr key={idx} className="hover:bg-slate-50">
                                                    <td className="p-4 font-mono">{user.phone}</td>
                                                    <td className="p-4">
                                                        <select 
                                                            value={user.active_model} 
                                                            onChange={(e) => updateUserField(user.phone, 'active_model', e.target.value)}
                                                            className="p-1 w-32"
                                                        >
                                                            {MODELS.map(m => <option key={m} value={m}>{m}</option>)}
                                                        </select>
                                                    </td>
                                                    <td className="p-4">
                                                        <input 
                                                            type="number" 
                                                            defaultValue={user.remaining_credits} 
                                                            onBlur={(e) => updateUserField(user.phone, 'remaining_credits', e.target.value)} 
                                                            className="w-16 p-1 text-center" 
                                                        />
                                                    </td>
                                                    <td className="p-4">
                                                         <select 
                                                            value={user.weather_city} 
                                                            onChange={(e) => updateUserField(user.phone, 'weather_city', e.target.value)}
                                                            className="p-1 w-24"
                                                        >
                                                            {CITIES.map(c => <option key={c} value={c}>{c}</option>)}
                                                        </select>
                                                    </td>
                                                     <td className="p-4">
                                                         <select 
                                                            value={user.zmanim_city} 
                                                            onChange={(e) => updateUserField(user.phone, 'zmanim_city', e.target.value)}
                                                            className="p-1 w-24"
                                                        >
                                                            {CITIES.map(c => <option key={c} value={c}>{c}</option>)}
                                                        </select>
                                                    </td>
                                                    <td className="p-4">
                                                        <button onClick={() => deleteUser(user.phone)} className="text-red-400 hover:text-red-600 p-2"><Icons.Trash /></button>
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        )}

                        {activeTab === 'guide' && <GuideTab />}
                        {activeTab === 'settings' && <SettingsTab config={config} onSaveConfig={saveConfig} />}

                    </div>
                </div>
            );
        }
        const root = ReactDOM.createRoot(document.getElementById('root'));
        root.render(<App />);
    </script>
</body>
</html>
"""

# ==============================================================================
#                              Backend Logic
# ==============================================================================

app = Flask(__name__)

# --- Helper Functions ---
def add_log(msg):
    timestamp = datetime.now().strftime('%H:%M:%S')
    print(f"[{timestamp}] {msg}")
    server_logs.append({"time": timestamp, "msg": str(msg)})
    if len(server_logs) > 50: server_logs.pop(0)

def load_json_file(filename, default_value={}):
    if not os.path.exists(filename): return default_value
    try:
        with open(filename, 'r', encoding='utf-8') as f: return json.load(f)
    except: return default_value

def save_json_file(filename, data):
    with open(filename, 'w', encoding='utf-8') as f: json.dump(data, f, ensure_ascii=False, indent=4)

def load_config():
    """טוען את ההגדרות הרגישות מהקובץ, אם קיים"""
    return load_json_file(CONFIG_FILE, {})

def is_system_configured():
    conf = load_config()
    required = ['yemot_username', 'yemot_password', 'gemini_api_key']
    return all(k in conf and conf[k] for k in required)

def read_credits_file():
    if not os.path.exists(CREDITS_FILE): return []
    with open(CREDITS_FILE, 'r', newline='', encoding='utf-8') as f: return list(csv.DictReader(f))

def write_credits_file(data):
    if not data: return
    # הגדרת עמודות קבועה ומלאה - מונע באגים של איבוד שדות
    fieldnames = ['phone', 'remaining_credits', 'initial_credits', 'date_added', 'active_model', 'zero_credit_notified', 'weather_city', 'zmanim_city']
    
    # וידוא שכל השורות מכילות את כל השדות לפני כתיבה
    cleaned_data = []
    for row in data:
        # השלמת שדות חסרים כדי למנוע מחיקתם או שגיאות
        for field in fieldnames:
            if field not in row or row[field] is None:
                if field == 'active_model': row[field] = DEFAULT_MODEL_NAME
                elif field == 'weather_city': row[field] = DEFAULT_WEATHER_CITY
                elif field == 'zmanim_city': row[field] = DEFAULT_ZMANIM_CITY
                else: row[field] = ''
        cleaned_data.append(row)

    with open(CREDITS_FILE, 'w', newline='', encoding='utf-8') as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames, extrasaction='ignore')
        writer.writeheader()
        writer.writerows(cleaned_data)

def init_files():
    # יצירה ראשונית או תיקון של קובץ הקרדיטים (Migration)
    fieldnames = ['phone', 'remaining_credits', 'initial_credits', 'date_added', 'active_model', 'zero_credit_notified', 'weather_city', 'zmanim_city']
    
    needs_rewrite = False
    data = []
    
    if os.path.exists(CREDITS_FILE):
        try:
            with open(CREDITS_FILE, 'r', newline='', encoding='utf-8') as f:
                reader = csv.DictReader(f)
                # בדיקה אם הכותרות בקובץ הקיים תואמות לחדשות
                if set(reader.fieldnames) != set(fieldnames):
                    needs_rewrite = True
                    data = list(reader)
        except: needs_rewrite = True
    else:
        needs_rewrite = True

    if needs_rewrite:
        print("מבצע עדכון מבנה קובץ נתונים (CSV Migration)...")
        # אם הקובץ לא קיים, data יהיה ריק והכותרות ייכתבו
        # אם הקובץ קיים, data יכיל את הנתונים הישנים ו-write_credits_file ישלים את החסר
        write_credits_file(data)

# --- Core Business Logic (User & Credits) ---
def get_user_settings(phone_number):
    """
    מחזיר את הגדרות המשתמש.
    תיקון: כעת בודק את קובץ ה-CSV גם עבור המנהל כדי לאפשר שמירת הגדרות (מודל/עיר).
    """
    conf = load_config()
    admin_phone = conf.get('admin_phone', '')
    
    # 1. חיפוש בקובץ המשתמשים הקיים
    users = read_credits_file()
    for user in users:
        if user['phone'] == phone_number:
            # מילוי ערכי ברירת מחדל אם חסרים
            if not user.get('active_model'): user['active_model'] = DEFAULT_MODEL_NAME
            if not user.get('weather_city'): user['weather_city'] = DEFAULT_WEATHER_CITY
            if not user.get('zmanim_city'): user['zmanim_city'] = DEFAULT_ZMANIM_CITY
            
            user['remaining_credits'] = float(user['remaining_credits'])
            return user
            
    # 2. אם המשתמש לא נמצא והוא המנהל - ניצור לו רשומה ב-CSV
    if admin_phone and phone_number == admin_phone:
        add_or_update_user(admin_phone, 999999) # יצירה עם קרדיט גבוה
        return get_user_settings(admin_phone) # קריאה חוזרת כדי לקבל את האובייקט מהקובץ
    
    return None

def use_credit(phone_number, amount=1):
    conf = load_config()
    # המנהל לא מאבד קרדיט בפועל, אבל הרשומה שלו קיימת ב-CSV לצרכי הגדרות
    if phone_number == conf.get('admin_phone'): return
    
    users = read_credits_file()
    for user in users:
        if user['phone'] == phone_number:
            user['remaining_credits'] = max(0, float(user['remaining_credits']) - amount)
            break
    write_credits_file(users)

def add_or_update_user(phone, amount):
    users = read_credits_file()
    found = False
    for user in users:
        if user['phone'] == phone:
            user['remaining_credits'] = amount
            user['initial_credits'] = amount
            found = True
            break
    if not found:
        users.append({
            'phone': phone, 'remaining_credits': amount, 'initial_credits': amount,
            'date_added': datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            'active_model': DEFAULT_MODEL_NAME, 'weather_city': DEFAULT_WEATHER_CITY,
            'zmanim_city': DEFAULT_ZMANIM_CITY, 'zero_credit_notified': '0'
        })
    write_credits_file(users)
    add_log(f"עודכן משתמש {phone} עם {amount} קרדיטים")

def update_single_field(phone, field, value):
    users = read_credits_file()
    found = False
    for user in users:
        if user['phone'] == phone:
            user[field] = value
            found = True
            break
    
    # אם מנסים לעדכן מנהל שעדיין לא קיים ב-CSV (מקרה קצה), ניצור אותו
    if not found:
        conf = load_config()
        if phone == conf.get('admin_phone'):
            add_or_update_user(phone, 999999)
            return update_single_field(phone, field, value) # נסה שוב
            
    write_credits_file(users)
    add_log(f"עודכן {field} ל-{value} עבור {phone}")

def get_user_history(phone):
    history_limit = load_config().get('history_limit', DEFAULT_HISTORY_LIMIT)
    return load_json_file(HISTORY_FILE).get(phone, [])[-int(history_limit):]

def save_user_history(phone, history):
    full = load_json_file(HISTORY_FILE)
    history_limit = load_config().get('history_limit', DEFAULT_HISTORY_LIMIT)
    full[phone] = history[-int(history_limit):]
    save_json_file(HISTORY_FILE, full)

# --- External API Handlers ---
def send_sms_via_yemot(phone, msg):
    conf = load_config()
    if not conf.get('yemot_username'): return False
    
    try:
        token = f"{conf['yemot_username']}:{conf['yemot_password']}"
        res = requests.get(f"{YEMOT_API_URL}/SendSms", params={"token": token, "phones": phone, "message": msg}, timeout=20)
        if res.json().get("responseStatus") == "OK":
            add_log(f"-> SMS נשלח ל-{phone}")
            return True
        add_log(f"שגיאת שליחה ל-{phone}: {res.text}")
    except Exception as e:
        add_log(f"שגיאת תקשורת שליחה: {e}")
    return False

def ask_gemini(prompt, history, model_name):
    conf = load_config()
    api_keys_str = conf.get('gemini_api_key', '')
    if not api_keys_str:
        return "שגיאת הגדרות: חסר מפתח API"
    
    # תמיכה ברשימת מפתחות (מופרדים בפסיק)
    api_keys = [k.strip() for k in api_keys_str.split(',') if k.strip()]
    if not api_keys:
        return "שגיאת הגדרות: רשימת מפתחות ריקה"

    model_id = GEMINI_MODELS.get(model_name, GEMINI_MODELS[DEFAULT_MODEL_NAME])
    system_inst_text = conf.get('prompt_suffix', DEFAULT_PROMPT_SUFFIX)
    
    payload = {
        "systemInstruction": {
            "parts": [{"text": system_inst_text}]
        },
        "contents": history + [{"role": "user", "parts": [{"text": prompt}]}]
    }
    
    last_error = None
    
    # לולאת רוטציה על המפתחות
    for i, current_key in enumerate(api_keys):
        url = f"https://generativelanguage.googleapis.com/v1beta/models/{model_id}:generateContent?key={current_key}"
        try:
            res = requests.post(url, json=payload, headers={'Content-Type': 'application/json'}, timeout=60)
            res.raise_for_status()
            return res.json()['candidates'][0]['content']['parts'][0]['text']
        except Exception as e:
            last_error = e
            add_log(f"מפתח API {i+1} נכשל: {e}. מנסה את הבא..." if i < len(api_keys)-1 else f"כל המפתחות נכשלו. שגיאה אחרונה: {e}")
            continue # נסה את המפתח הבא
            
    return "שגיאה בתקשורת עם ה-AI (כל המפתחות נכשלו)."

def parse_iso_time(iso_str):
    """ממיר מחרוזת זמן ISO לשעה בפורמט HH:MM"""
    if not iso_str: return "לא זמין"
    try:
        # טיפול בפורמטים שונים של Hebcal
        if 'T' in iso_str:
            dt = datetime.fromisoformat(iso_str)
            return dt.strftime('%H:%M')
        return iso_str[:5]
    except:
        return iso_str[:5]

def get_zmanim(city):
    """
    מחזיר את זמני היום בהתאם להגדרות המשתמש.
    קורא מה-Config אלו זמנים להציג.
    """
    geo = ZMANIM_CITIES.get(city)
    if not geo: return "עיר לא ידועה"
    
    try:
        res = requests.get(f"{ZMANIM_API_URL_BASE}&geonameid={geo}", timeout=10).json()
        times = res.get('times', {})
        
        # טעינת הגדרות הזמנים להצגה
        conf = load_config()
        enabled_zmanim = conf.get('zmanim_settings', DEFAULT_ENABLED_ZMANIM)
        
        output_lines = [f"זמני היום ({city}):"]
        
        # מעבר על כל האפשרויות לפי הסדר שהגדרנו, ובדיקה אם הן מסומנות להצגה
        for opt in ALL_ZMANIM_OPTIONS:
            if opt['id'] in enabled_zmanim:
                time_val = times.get(opt['hebcal_key'])
                if time_val:
                    formatted_time = parse_iso_time(time_val)
                    output_lines.append(f"{opt['label']}: {formatted_time}")
        
        if len(output_lines) == 1: # אם לא נבחרו זמנים או לא נמצאו
            return f"זמני היום ({city}):\nלא נבחרו זמנים להצגה בהגדרות."
            
        return "\n".join(output_lines)

    except Exception as e:
        print(f"Zmanim Error: {e}") 
        return "שגיאה בקבלת זמנים"

def get_weather_forecast(city_name):
    coords = WEATHER_CITIES.get(city_name)
    if not coords: return "עיר לא ידועה."
    forecast_text = f"מזג אוויר - {city_name}:\n"
    try:
        params = {'latitude': coords['lat'], 'longitude': coords['lon'], 'daily': 'temperature_2m_max,temperature_2m_min', 'forecast_days': 3}
        response = requests.get(WEATHER_API_URL, params=params, timeout=10)
        response.raise_for_status()
        data = response.json().get('daily', {})
        if 'time' in data and len(data['time']) >= 3:
            forecast_text += f"היום: {int(data['temperature_2m_min'][0])}-{int(data['temperature_2m_max'][0])}°\n"
            forecast_text += f"מחר: {int(data['temperature_2m_min'][1])}-{int(data['temperature_2m_max'][1])}°\n"
            forecast_text += f"מחרתיים: {int(data['temperature_2m_min'][2])}-{int(data['temperature_2m_max'][2])}°"
            return forecast_text.strip()
        return "שגיאה בעיבוד נתוני מזג אוויר."
    except: return "שגיאה בקבלת מזג אוויר"

def get_dollar_rate(amount=1.0):
    try:
        res = requests.get(CURRENCY_API_URL, timeout=10).json()
        rate = res.get('rates', {}).get('ILS')
        if rate:
            if amount == 1.0: return f"שער הדולר: {rate:.3f} ₪"
            return f"{amount} דולר = {amount * rate:.2f} ₪"
    except: pass
    return "שגיאה בקבלת שער דולר"

# --- Bot Loop Logic ---

def handle_stateful_reply(sender, content):
    state = USER_STATES.get(sender)
    if not state: return False
    
    action = state.get('action')
    try:
        idx = int(content.strip()) - 1
        
        if action == 'awaiting_model_choice':
            models = list(GEMINI_MODELS.keys())
            if 0 <= idx < len(models):
                update_single_field(sender, 'active_model', models[idx])
                send_sms_via_yemot(sender, f"המודל שונה בהצלחה ל: {models[idx]}")
            else: send_sms_via_yemot(sender, "בחירה לא חוקית")
            
        elif action == 'awaiting_weather_city_choice':
            cities = list(WEATHER_CITIES.keys())
            if 0 <= idx < len(cities):
                update_single_field(sender, 'weather_city', cities[idx])
                send_sms_via_yemot(sender, f"עיר מזג אוויר שונתה ל: {cities[idx]}")
            else: send_sms_via_yemot(sender, "בחירה לא חוקית")
            
        elif action == 'awaiting_zmanim_city_choice':
            cities = list(ZMANIM_CITIES.keys())
            if 0 <= idx < len(cities):
                update_single_field(sender, 'zmanim_city', cities[idx])
                send_sms_via_yemot(sender, f"עיר זמנים שונתה ל: {cities[idx]}")
            else: send_sms_via_yemot(sender, "בחירה לא חוקית")
            
        del USER_STATES[sender]
        return True
    except:
        send_sms_via_yemot(sender, "נא לשלוח מספר תקין")
        return True

def parse_command(sender, content):
    cmd = content.strip().lower()
    
    # המרת דולר (זיהוי עם RegEx)
    match = re.search(r'(כמה זה\s*)?(\d+\.?\d*)\s+(דולר|דולרים|usd)', cmd)
    if match: return {'type': 'dollar', 'amount': float(match.group(2))}
    
    if cmd in ["דולר", "שער הדולר"]: return {'type': 'dollar', 'amount': 1.0}
    if cmd == "מזג אוויר": return {'type': 'weather'}
    if cmd in ["זמני היום", "זמנים"]: return {'type': 'zmanim'}
    if cmd == "סטטוס": return {'type': 'status'}
    if cmd == "איפוס": return {'type': 'reset'}
    if cmd == "שינוי מודל": return {'type': 'menu_model'}
    if cmd == "שינוי אזור": return {'type': 'menu_weather'}
    if cmd == "שינוי אזור זמנים": return {'type': 'menu_zmanim'}
    
    return None

def process_message(sms):
    sender = sms.get('source', '').replace('-', '')
    content = sms.get('message', '').strip()
    
    # 1. בדיקה אם המשתמש באמצע תהליך (State)
    if handle_stateful_reply(sender, content):
        return

    user = get_user_settings(sender)
    if not user:
        add_log(f"הודעה ממשתמש לא רשום {sender}. מתעלם.")
        return

    # 2. בדיקה אם זו פקודה מיוחדת
    command = parse_command(sender, content)
    
    if command:
        if user['remaining_credits'] <= 0:
            send_sms_via_yemot(sender, "אין לך מספיק קרדיט.")
            return

        cmd_type = command['type']
        
        if cmd_type == 'dollar':
            send_sms_via_yemot(sender, get_dollar_rate(command['amount']))
        elif cmd_type == 'weather':
            send_sms_via_yemot(sender, get_weather_forecast(user['weather_city']))
        elif cmd_type == 'zmanim':
            send_sms_via_yemot(sender, get_zmanim(user['zmanim_city']))
        elif cmd_type == 'status':
            send_sms_via_yemot(sender, f"מודל: {user['active_model']}\nיתרה: {user['remaining_credits']}")
        elif cmd_type == 'reset':
            save_user_history(sender, [])
            send_sms_via_yemot(sender, "ההיסטוריה אופסה.")
        elif cmd_type == 'menu_model':
            menu = "בחר מודל:\n" + "\n".join([f"{i+1}. {m}" for i, m in enumerate(GEMINI_MODELS.keys())])
            USER_STATES[sender] = {'action': 'awaiting_model_choice'}
            send_sms_via_yemot(sender, menu)
        elif cmd_type == 'menu_weather':
            menu = "בחר עיר למזג אוויר:\n" + "\n".join([f"{i+1}. {c}" for i, c in enumerate(WEATHER_CITIES.keys())])
            USER_STATES[sender] = {'action': 'awaiting_weather_city_choice'}
            send_sms_via_yemot(sender, menu)
        elif cmd_type == 'menu_zmanim':
            menu = "בחר עיר לזמנים:\n" + "\n".join([f"{i+1}. {c}" for i, c in enumerate(ZMANIM_CITIES.keys())])
            USER_STATES[sender] = {'action': 'awaiting_zmanim_city_choice'}
            send_sms_via_yemot(sender, menu)
            
        use_credit(sender, 1)
        return

    # 3. אם לא פקודה, זו שיחה עם ה-AI
    if user['remaining_credits'] <= 0:
        send_sms_via_yemot(sender, "נגמר הקרדיט.")
        return

    add_log(f"הודעה מ-{sender}: {content[:20]}...")
    history = get_user_history(sender)
    reply = ask_gemini(content, history, user['active_model'])
    
    if len(reply) > MAX_RESPONSE_LENGTH: reply = reply[:MAX_RESPONSE_LENGTH]
    
    send_sms_via_yemot(sender, reply)
    use_credit(sender, 1)
    
    history.append({"role": "user", "parts": [{"text": content}]})
    history.append({"role": "model", "parts": [{"text": reply}]})
    save_user_history(sender, history)

def bot_loop():
    while bot_active:
        if not is_system_configured():
            time.sleep(2)
            continue
            
        conf = load_config()
        token = f"{conf['yemot_username']}:{conf['yemot_password']}"
        
        try:
            res = requests.get(f"{YEMOT_API_URL}/GetIncomingSms", params={'token': token}, timeout=30)
            if res.ok and res.json().get("responseStatus") == "OK":
                msgs = res.json().get('rows', [])
                for sms in msgs:
                    sms_id = f"{sms['source']}-{sms['receive_date']}-{hash(sms['message'])}"
                    if sms_id in PROCESSED_SMS_IDS: continue
                    
                    try:
                        msg_time = datetime.strptime(sms['receive_date'], '%Y-%m-%d %H:%M:%S')
                        if msg_time < APP_START_TIME: continue
                    except: pass
                    
                    PROCESSED_SMS_IDS.add(sms_id)
                    process_message(sms)
        except Exception as e:
            add_log(f"Loop Error: {e}")
        time.sleep(POLLING_INTERVAL_SECONDS)

# --- Flask Routes ---
@app.route('/')
def home():
    return render_template_string(HTML_TEMPLATE)

@app.route('/api/status')
def api_status():
    return jsonify({"configured": is_system_configured()})

@app.route('/api/setup', methods=['POST'])
def api_setup():
    data = request.json
    save_json_file(CONFIG_FILE, data)
    return jsonify({"status": "ok"})

@app.route('/api/reset_config', methods=['POST'])
def api_reset_config():
    if os.path.exists(CONFIG_FILE):
        os.remove(CONFIG_FILE)
    return jsonify({"status": "ok"})

@app.route('/api/config')
def api_get_config():
    """החזרת הקונפיגורציה (ללא סיסמאות) עבור דף ההגדרות"""
    full_conf = load_config()
    safe_conf = {k: v for k, v in full_conf.items() if k not in ['yemot_password', 'gemini_api_key']}
    # הוספת ערך ברירת המחדל כדי שיוצג באתר
    safe_conf['default_prompt_suffix'] = DEFAULT_PROMPT_SUFFIX
    return jsonify(safe_conf)

@app.route('/api/update_config', methods=['POST'])
def api_update_config():
    """עדכון הגדרות כלליות (כמו הנחיית מערכת)"""
    new_data = request.json
    current_conf = load_config()
    # עדכון רק של שדות מותרים, שמירה על סיסמאות קיימות
    for k, v in new_data.items():
        if k in ['prompt_suffix', 'zmanim_settings', 'history_limit']: # רשימת שדות מותרים לעריכה
            current_conf[k] = v
    save_json_file(CONFIG_FILE, current_conf)
    return jsonify({"status": "ok"})

@app.route('/api/users')
def api_users():
    if not is_system_configured(): return jsonify([])
    return jsonify(read_credits_file())

@app.route('/api/logs')
def api_logs(): return jsonify(server_logs)

@app.route('/api/stats')
def api_stats():
    status = "פעיל" if is_system_configured() else "ממתין להגדרה"
    return jsonify({"total_users": len(read_credits_file()), "total_requests": len(PROCESSED_SMS_IDS), "balance": status})

@app.route('/api/update_user', methods=['POST'])
def api_update():
    d = request.json
    if 'field' in d: # עדכון שדה ספציפי
        update_single_field(d['phone'], d['field'], d['value'])
    else: # תמיכה לאחור
        add_or_update_user(d['phone'], d['credit'])
    return jsonify({"status": "ok"})

@app.route('/api/delete_user', methods=['POST'])
def api_delete():
    d = request.json
    users = [u for u in read_credits_file() if u['phone'] != d['phone']]
    write_credits_file(users)
    return jsonify({"status": "ok"})

if __name__ == '__main__':
    init_files()
    threading.Thread(target=bot_loop, daemon=True).start()
    app.run(host='0.0.0.0', port=5000, debug=False)