// --- משתנים גלובליים ---
let chats = [];
// הגדרת תיקיות ברירת מחדל
const DEFAULT_FOLDERS = [
  { id: 'programming', name: 'תכנות' },
  { id: 'writing', name: 'כתיבה' },
  { id: 'general', name: 'כללי' }
];

let folders = []; 
let currentUrl = window.location.href;
let isCollapsed = false; 
let currentFolderId = 'all'; // 'all' מציג את הכל
const SIDEBAR_WIDTH = '300px';

// --- לוגיקה של ממשק (Layout) ---

function adjustPageLayout(collapsed) {
  const body = document.body;
  body.style.transition = 'margin-right 0.3s cubic-bezier(0.4, 0.0, 0.2, 1)';
  body.style.marginRight = collapsed ? '0px' : SIDEBAR_WIDTH;
}

function createInterface() {
  const existingSidebar = document.getElementById('custom-gemini-sidebar');
  const existingBtn = document.getElementById('ghk-toggle-btn');
  if (existingSidebar) existingSidebar.remove();
  if (existingBtn) existingBtn.remove();

  // כפתור צף
  const toggleBtn = document.createElement('button');
  toggleBtn.id = 'ghk-toggle-btn';
  toggleBtn.title = 'היסטוריית שיחות';
  toggleBtn.innerHTML = getArrowIcon(false);
  document.body.appendChild(toggleBtn);

  // סרגל צד
  const sidebar = document.createElement('div');
  sidebar.id = 'custom-gemini-sidebar';
  sidebar.innerHTML = `
    <div class="ghk-header">
      <h3>השיחות שלי</h3>
      <div class="ghk-folder-controls">
        <select id="ghk-folder-select">
          <option value="all">הצג הכל</option>
          <!-- תיקיות יטענו לכאן -->
        </select>
        <button id="ghk-new-folder-btn" title="צור תיקייה חדשה">+</button>
      </div>
    </div>
    
    <div class="ghk-controls">
      <button id="ghk-add-btn">
        <span>+</span> שמור בתיקייה הנוכחית
      </button>
    </div>
    
    <div id="ghk-list" class="ghk-list"></div>
  `;
  document.body.appendChild(sidebar);

  // שחזור מצב שמור
  chrome.storage.local.get(['ghkCollapsed'], function(result) {
    setSidebarState(result.ghkCollapsed === true);
  });

  // האזנות
  toggleBtn.addEventListener('click', toggleSidebar);
  document.getElementById('ghk-add-btn').addEventListener('click', saveCurrentChat);
  document.getElementById('ghk-new-folder-btn').addEventListener('click', createNewFolder);
  
  const folderSelect = document.getElementById('ghk-folder-select');
  folderSelect.addEventListener('change', (e) => {
    currentFolderId = e.target.value;
    renderList();
  });
}

function setSidebarState(collapsed) {
  const sidebar = document.getElementById('custom-gemini-sidebar');
  const btn = document.getElementById('ghk-toggle-btn');
  
  isCollapsed = collapsed;
  
  if (collapsed) {
    sidebar.classList.add('ghk-collapsed');
    btn.classList.add('ghk-collapsed');
    btn.innerHTML = getArrowIcon(true);
  } else {
    sidebar.classList.remove('ghk-collapsed');
    btn.classList.remove('ghk-collapsed');
    btn.innerHTML = getArrowIcon(false);
  }
  adjustPageLayout(collapsed);
  chrome.storage.local.set({ ghkCollapsed: collapsed });
}

function toggleSidebar() { setSidebarState(!isCollapsed); }

function getArrowIcon(isClosed) {
  return isClosed 
    ? '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M15 18l-6-6 6-6"/></svg>'
    : '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M9 18l6-6-6-6"/></svg>';
}

// --- לוגיקה של נתונים (Data) ---

function loadData() {
  chrome.storage.local.get(['geminiChats', 'geminiFolders'], function(result) {
    // טעינת תיקיות או אתחול ברירת מחדל
    if (result.geminiFolders && result.geminiFolders.length > 0) {
      folders = result.geminiFolders;
    } else {
      // אם אין תיקיות בכלל (פעם ראשונה או שדרוג גרסה), ניצור את ברירת המחדל
      folders = [...DEFAULT_FOLDERS];
      saveData(); // שומר מיד כדי שיהיה לפעם הבאה
    }

    if (result.geminiChats) {
      chats = result.geminiChats;
      // מיגרציה לנתונים ישנים
      chats = chats.map(c => ({
        ...c,
        timestamp: c.timestamp || Date.now(),
        folderId: c.folderId || 'general', // אם לא מוגדר, הולך לכללי
        isPinned: c.isPinned || false
      }));
    }
    renderFolderSelectOptions();
    renderList();
  });
}

function saveData() {
  chrome.storage.local.set({ geminiChats: chats, geminiFolders: folders });
}

// --- לוגיקה של תיקיות ---

function createNewFolder() {
  const name = prompt('שם התיקייה החדשה:');
  if (name && name.trim()) {
    const newFolder = {
      id: 'folder_' + Date.now(),
      name: name.trim()
    };
    folders.push(newFolder);
    saveData();
    renderFolderSelectOptions();
    
    // מעבר אוטומטי לתיקייה החדשה
    currentFolderId = newFolder.id;
    document.getElementById('ghk-folder-select').value = newFolder.id;
    renderList();
  }
}

function renderFolderSelectOptions() {
  const select = document.getElementById('ghk-folder-select');
  const savedValue = select.value || 'all';
  
  select.innerHTML = '<option value="all">הצג הכל</option>';
  
  folders.forEach(f => {
    const option = document.createElement('option');
    option.value = f.id;
    option.textContent = f.name;
    select.appendChild(option);
  });
  
  if (folders.some(f => f.id === savedValue) || savedValue === 'all') {
    select.value = savedValue;
    currentFolderId = savedValue;
  } else {
    select.value = 'all';
    currentFolderId = 'all';
  }
}

// --- לוגיקה של שיחות ---

function saveCurrentChat() {
  const url = window.location.href;
  if (chats.some(c => c.url === url)) {
    alert('השיחה הזו כבר שמורה!');
    return;
  }

  let suggestedTitle = document.title.replace(/Gemini/g, '').replace(/- Google/g, '').trim() || "שיחה חדשה";
  const title = prompt('שם לשיחה:', suggestedTitle);
  
  if (title) {
    // אם אנחנו ב"הצג הכל", שומרים ל"כללי", אחרת לתיקייה הפתוחה
    const targetFolderId = (currentFolderId === 'all') ? 'general' : currentFolderId;
    
    // בדיקה אם התיקייה קיימת (למקרה שנמחקה ידנית), אם לא - כללי
    const folderExists = folders.some(f => f.id === targetFolderId);
    const finalFolderId = folderExists ? targetFolderId : 'general';
    const folderName = folders.find(f => f.id === finalFolderId)?.name || 'כללי';

    const newChat = {
      id: Date.now(),
      title: title,
      url: url,
      timestamp: Date.now(),
      folderId: finalFolderId,
      isPinned: false
    };
    
    chats.unshift(newChat);
    saveData();
    renderList();
    
    const btn = document.getElementById('ghk-add-btn');
    btn.innerHTML = `נשמר ב"${folderName}"!`;
    setTimeout(() => {
       btn.innerHTML = `<span>+</span> שמור בתיקייה הנוכחית`;
    }, 2000);
  }
}

// פונקציה חדשה לשינוי שם
function renameChat(chatId) {
  const chat = chats.find(c => c.id === chatId);
  if (!chat) return;

  const newName = prompt('ערוך שם שיחה:', chat.title);
  if (newName && newName.trim()) {
    chat.title = newName.trim();
    saveData();
    renderList();
  }
}

function togglePin(chatId) {
  const chat = chats.find(c => c.id === chatId);
  if (chat) {
    chat.isPinned = !chat.isPinned;
    saveData();
    renderList();
  }
}

function deleteChat(id) {
  if (confirm('למחוק את השיחה?')) {
    chats = chats.filter(c => c.id !== id);
    saveData();
    renderList();
  }
}

function formatTime(timestamp) {
  if (!timestamp) return '';
  const date = new Date(timestamp);
  return date.toLocaleDateString('he-IL', {
    day: '2-digit', month: '2-digit'
  }) + ' ' + date.toLocaleTimeString('he-IL', {
    hour: '2-digit', minute: '2-digit'
  });
}

function renderList() {
  const listEl = document.getElementById('ghk-list');
  if (!listEl) return;
  listEl.innerHTML = '';

  let filteredChats = chats;
  if (currentFolderId !== 'all') {
    filteredChats = chats.filter(c => c.folderId === currentFolderId);
  }

  filteredChats.sort((a, b) => {
    if (a.isPinned === b.isPinned) {
      return b.timestamp - a.timestamp;
    }
    return a.isPinned ? -1 : 1;
  });

  if (filteredChats.length === 0) {
    listEl.innerHTML = '<div style="text-align:center; color:#999; padding:20px; font-size:12px;">אין שיחות בתיקייה זו</div>';
    return;
  }

  // SVG לאייקונים
  const pinIcon = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="12" y1="17" x2="12" y2="22"></line><path d="M5 17h14v-1.76a2 2 0 0 0-1.11-1.79l-1.78-.9A2 2 0 0 1 15 10.76V6h1a2 2 0 0 0 0-4H8a2 2 0 0 0 0 4h1v4.76a2 2 0 0 1-1.11 1.79l-1.78.9A2 2 0 0 0 5 15.24Z"></path></svg>`;
  const editIcon = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path></svg>`;

  filteredChats.forEach(chat => {
    const item = document.createElement('div');
    item.className = 'ghk-item';
    if (window.location.href.includes(chat.url)) item.classList.add('active');
    if (chat.isPinned) item.classList.add('pinned');
    
    item.innerHTML = `
      <div class="ghk-item-content" title="${chat.title}">
        <div class="ghk-title">${chat.title}</div>
        <div class="ghk-meta">
          <span>${formatTime(chat.timestamp)}</span>
        </div>
      </div>
      <div class="ghk-actions">
        <button class="ghk-action-btn ghk-edit-btn" data-id="${chat.id}" title="שנה שם">
          ${editIcon}
        </button>
        <button class="ghk-action-btn ghk-pin-btn ${chat.isPinned ? 'active' : ''}" data-id="${chat.id}" title="${chat.isPinned ? 'בטל הצמדה' : 'הצמד להתחלה'}">
          ${pinIcon}
        </button>
        <button class="ghk-action-btn ghk-del-btn" data-id="${chat.id}" title="מחק">×</button>
      </div>
    `;

    item.addEventListener('click', (e) => {
        if (e.target.closest('.ghk-action-btn')) return;
        window.location.href = chat.url;
    });

    // הוספת האזנה לכפתור העריכה החדש
    item.querySelector('.ghk-edit-btn').addEventListener('click', (e) => {
      e.stopPropagation();
      renameChat(chat.id);
    });

    item.querySelector('.ghk-pin-btn').addEventListener('click', (e) => {
      e.stopPropagation();
      togglePin(chat.id);
    });

    item.querySelector('.ghk-del-btn').addEventListener('click', (e) => {
      e.stopPropagation();
      deleteChat(chat.id);
    });

    listEl.appendChild(item);
  });
}

// בדיקת שינוי URL
setInterval(() => {
  if (window.location.href !== currentUrl) {
    currentUrl = window.location.href;
    renderList();
  }
}, 1000);

// אתחול
setTimeout(() => {
  createInterface();
  loadData();
}, 1000);