// Check authentication status on load
document.addEventListener("DOMContentLoaded", async () => {
  const status = await checkAuthStatus()
  updateUI(status)

  // Load last code if available
  const lastCode = await getLastCode()
  if (lastCode) {
    displayLastCode(lastCode)
  }
})

// Connect button
document.getElementById("connectBtn").addEventListener("click", async () => {
  try {
    const token = await window.chrome.identity.getAuthToken({ interactive: true })
    if (token) {
      await window.chrome.storage.local.set({ authToken: token, isAuthenticated: true })
      updateUI(true)

      // Start monitoring
      window.chrome.runtime.sendMessage({ action: "startMonitoring" })
    }
  } catch (error) {
    console.error("Authentication failed:", error)
    alert("שגיאה בהתחברות. אנא נסה שוב.")
  }
})

// Disconnect button
document.getElementById("disconnectBtn").addEventListener("click", async () => {
  try {
    const token = await window.chrome.storage.local.get("authToken")
    if (token.authToken) {
      await window.chrome.identity.removeCachedAuthToken({ token: token.authToken })
    }
    await window.chrome.storage.local.remove(["authToken", "isAuthenticated"])
    updateUI(false)

    // Stop monitoring
    window.chrome.runtime.sendMessage({ action: "stopMonitoring" })
  } catch (error) {
    console.error("Disconnect failed:", error)
  }
})

// Settings button
document.getElementById("settingsBtn").addEventListener("click", () => {
  window.chrome.runtime.openOptionsPage()
})

async function checkAuthStatus() {
  const result = await window.chrome.storage.local.get("isAuthenticated")
  return result.isAuthenticated || false
}

async function getLastCode() {
  const result = await window.chrome.storage.local.get("lastOtpCode")
  return result.lastOtpCode || null
}

function displayLastCode(code) {
  document.getElementById("lastCode").style.display = "block"
  document.getElementById("codeDisplay").textContent = code
}

function updateUI(isAuthenticated) {
  const statusDiv = document.getElementById("status")
  const connectBtn = document.getElementById("connectBtn")
  const disconnectBtn = document.getElementById("disconnectBtn")

  if (isAuthenticated) {
    statusDiv.className = "status connected"
    statusDiv.textContent = "✓ מחובר ל-Gmail - מנטר מיילים"
    connectBtn.style.display = "none"
    disconnectBtn.style.display = "block"
  } else {
    statusDiv.className = "status disconnected"
    statusDiv.textContent = "לא מחובר ל-Gmail"
    connectBtn.style.display = "block"
    disconnectBtn.style.display = "none"
  }
}

// Listen for OTP updates
window.chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "otpFound") {
    displayLastCode(message.code)

    // Update status to show monitoring
    const statusDiv = document.getElementById("status")
    statusDiv.className = "status monitoring"
    statusDiv.textContent = "🔍 קוד נמצא ומולא!"

    setTimeout(() => {
      statusDiv.className = "status connected"
      statusDiv.textContent = "✓ מחובר ל-Gmail - מנטר מיילים"
    }, 3000)
  }
})
