// Load settings on page load
document.addEventListener("DOMContentLoaded", async () => {
  await loadSettings()
  await loadRules()
})

// Add rule button
document.getElementById("addRuleBtn").addEventListener("click", async () => {
  const websiteUrl = document.getElementById("websiteUrl").value.trim()
  const senderEmail = document.getElementById("senderEmail").value.trim()
  const otpPattern = document.getElementById("otpPattern").value.trim() || "\\d{6}"

  if (!websiteUrl || !senderEmail) {
    alert("אנא מלא את כתובת האתר וכתובת המייל")
    return
  }

  // Validate URL
  try {
    new URL(websiteUrl)
  } catch (e) {
    alert("כתובת האתר לא תקינה. אנא הכנס URL מלא (לדוגמה: https://www.example.com)")
    return
  }

  // Validate email
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
  if (!emailRegex.test(senderEmail)) {
    alert("כתובת המייל לא תקינה")
    return
  }

  // Add rule
  const rules = await getRules()
  rules.push({
    id: Date.now(),
    websiteUrl,
    senderEmail,
    otpPattern,
    enabled: true,
  })

  const chrome = window.chrome // Declare chrome variable
  await chrome.storage.local.set({ otpRules: rules })

  // Clear form
  document.getElementById("websiteUrl").value = ""
  document.getElementById("senderEmail").value = ""
  document.getElementById("otpPattern").value = ""

  // Reload rules list
  await loadRules()

  showSuccessMessage()
})

// Save settings button
document.getElementById("saveBtn").addEventListener("click", async () => {
  const checkInterval = Number.parseInt(document.getElementById("checkInterval").value) || 30
  const autoFillEnabled = document.getElementById("autoFillEnabled").checked
  const notificationsEnabled = document.getElementById("notificationsEnabled").checked

  if (checkInterval < 10) {
    alert("תדירות הבדיקה חייבת להיות לפחות 10 שניות")
    return
  }

  const chrome = window.chrome // Declare chrome variable
  await chrome.storage.local.set({
    checkInterval: checkInterval * 1000, // Convert to milliseconds
    autoFillEnabled,
    notificationsEnabled,
  })

  showSuccessMessage()

  // Restart monitoring with new settings
  chrome.runtime.sendMessage({ action: "restartMonitoring" })
})

// Reset button
document.getElementById("resetBtn").addEventListener("click", async () => {
  if (confirm("האם אתה בטוח שברצונך לאפס את כל ההגדרות?")) {
    const chrome = window.chrome // Declare chrome variable
    await chrome.storage.local.clear()
    location.reload()
  }
})

async function loadSettings() {
  const settings = await window.chrome.storage.local.get(["checkInterval", "autoFillEnabled", "notificationsEnabled"])

  document.getElementById("checkInterval").value = (settings.checkInterval || 30000) / 1000
  document.getElementById("autoFillEnabled").checked = settings.autoFillEnabled !== false
  document.getElementById("notificationsEnabled").checked = settings.notificationsEnabled !== false
}

async function loadRules() {
  const rules = await getRules()
  const rulesList = document.getElementById("rulesList")

  if (rules.length === 0) {
    rulesList.innerHTML = `
      <p style="color: #999; text-align: center; padding: 20px;">
        עדיין לא הוספת כללים. הוסף כלל ראשון למעלה!
      </p>
    `
    return
  }

  rulesList.innerHTML = rules
    .map(
      (rule) => `
    <div class="rule-item" data-id="${rule.id}">
      <div class="rule-header">
        <span class="rule-domain">${new URL(rule.websiteUrl).hostname}</span>
        <button class="btn-remove" onclick="removeRule(${rule.id})">🗑️ מחק</button>
      </div>
      <div style="font-size: 13px; color: #666;">
        <div><strong>אתר:</strong> ${rule.websiteUrl}</div>
        <div><strong>מייל:</strong> ${rule.senderEmail}</div>
        <div><strong>תבנית:</strong> <code>${rule.otpPattern}</code></div>
      </div>
    </div>
  `,
    )
    .join("")
}

async function getRules() {
  const result = await window.chrome.storage.local.get("otpRules")
  return result.otpRules || []
}

window.removeRule = async (ruleId) => {
  if (!confirm("האם אתה בטוח שברצונך למחוק כלל זה?")) {
    return
  }

  const rules = await getRules()
  const updatedRules = rules.filter((rule) => rule.id !== ruleId)
  const chrome = window.chrome // Declare chrome variable
  await chrome.storage.local.set({ otpRules: updatedRules })
  await loadRules()
  showSuccessMessage()
}

function showSuccessMessage() {
  const message = document.getElementById("successMessage")
  message.style.display = "block"
  setTimeout(() => {
    message.style.display = "none"
  }, 3000)
}
