// Declare the chrome variable
const chrome = window.chrome

let monitoringInterval = null

// Listen for messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "startMonitoring") {
    startMonitoring()
  } else if (message.action === "stopMonitoring") {
    stopMonitoring()
  } else if (message.action === "restartMonitoring") {
    stopMonitoring()
    startMonitoring()
  }
})

// Start monitoring on extension install/update
chrome.runtime.onInstalled.addListener(() => {
  console.log("[v0] OTP Auto-Fill extension installed")
  checkAndStartMonitoring()
})

// Start monitoring on browser startup
chrome.runtime.onStartup.addListener(() => {
  console.log("[v0] Browser started, checking monitoring status")
  checkAndStartMonitoring()
})

async function checkAndStartMonitoring() {
  const { isAuthenticated } = await chrome.storage.local.get("isAuthenticated")
  if (isAuthenticated) {
    startMonitoring()
  }
}

async function startMonitoring() {
  console.log("[v0] Starting email monitoring")

  // Stop existing monitoring if any
  stopMonitoring()

  // Get check interval from settings
  const { checkInterval } = await chrome.storage.local.get("checkInterval")
  const interval = checkInterval || 30000 // Default 30 seconds

  // Check immediately
  checkForOTP()

  // Then check periodically
  monitoringInterval = setInterval(checkForOTP, interval)
}

function stopMonitoring() {
  console.log("[v0] Stopping email monitoring")
  if (monitoringInterval) {
    clearInterval(monitoringInterval)
    monitoringInterval = null
  }
}

async function checkForOTP() {
  try {
    console.log("[v0] Checking for OTP emails...")

    // Get auth token
    const { authToken } = await chrome.storage.local.get("authToken")
    if (!authToken) {
      console.log("[v0] No auth token found")
      return
    }

    // Get rules
    const { otpRules } = await chrome.storage.local.get("otpRules")
    if (!otpRules || otpRules.length === 0) {
      console.log("[v0] No rules configured")
      return
    }

    // Check each rule
    for (const rule of otpRules) {
      if (!rule.enabled) continue

      await checkRuleForOTP(authToken, rule)
    }
  } catch (error) {
    console.error("[v0] Error checking for OTP:", error)

    // If token is invalid, clear authentication
    if (error.message && error.message.includes("401")) {
      await chrome.storage.local.remove(["authToken", "isAuthenticated"])
    }
  }
}

async function checkRuleForOTP(authToken, rule) {
  try {
    // Search for recent emails from the sender
    const query = `from:${rule.senderEmail} newer_than:5m`
    const searchUrl = `https://gmail.googleapis.com/gmail/v1/users/me/messages?q=${encodeURIComponent(query)}&maxResults=5`

    const searchResponse = await fetch(searchUrl, {
      headers: {
        Authorization: `Bearer ${authToken}`,
      },
    })

    if (!searchResponse.ok) {
      throw new Error(`Gmail API error: ${searchResponse.status}`)
    }

    const searchData = await searchResponse.json()

    if (!searchData.messages || searchData.messages.length === 0) {
      console.log(`[v0] No recent emails from ${rule.senderEmail}`)
      return
    }

    // Get the most recent message
    const messageId = searchData.messages[0].id
    const messageUrl = `https://gmail.googleapis.com/gmail/v1/users/me/messages/${messageId}`

    const messageResponse = await fetch(messageUrl, {
      headers: {
        Authorization: `Bearer ${authToken}`,
      },
    })

    const messageData = await messageResponse.json()

    // Extract email body
    let emailBody = ""
    if (messageData.payload.body.data) {
      emailBody = atob(messageData.payload.body.data.replace(/-/g, "+").replace(/_/g, "/"))
    } else if (messageData.payload.parts) {
      for (const part of messageData.payload.parts) {
        if (part.mimeType === "text/plain" && part.body.data) {
          emailBody += atob(part.body.data.replace(/-/g, "+").replace(/_/g, "/"))
        }
      }
    }

    console.log("[v0] Email body:", emailBody.substring(0, 200))

    // Extract OTP code using regex
    const otpRegex = new RegExp(rule.otpPattern)
    const match = emailBody.match(otpRegex)

    if (match) {
      const otpCode = match[0]
      console.log("[v0] OTP code found:", otpCode)

      // Save the code
      await chrome.storage.local.set({
        lastOtpCode: otpCode,
        lastOtpTimestamp: Date.now(),
        lastOtpRule: rule,
      })

      // Show notification
      const { notificationsEnabled } = await chrome.storage.local.get("notificationsEnabled")
      if (notificationsEnabled !== false) {
        chrome.notifications.create({
          type: "basic",
          iconUrl: "icons/icon128.jpg",
          title: "קוד אימות נמצא!",
          message: `הקוד ${otpCode} נמצא עבור ${new URL(rule.websiteUrl).hostname}`,
          priority: 2,
        })
      }

      // Try to fill the code in the matching website
      await fillOTPInWebsite(rule.websiteUrl, otpCode)

      // Notify popup
      chrome.runtime.sendMessage({
        action: "otpFound",
        code: otpCode,
        rule: rule,
      })
    }
  } catch (error) {
    console.error("[v0] Error checking rule:", error)
  }
}

async function fillOTPInWebsite(websiteUrl, otpCode) {
  try {
    const { autoFillEnabled } = await chrome.storage.local.get("autoFillEnabled")
    if (autoFillEnabled === false) {
      console.log("[v0] Auto-fill is disabled")
      return
    }

    // Find tabs matching the website
    const url = new URL(websiteUrl)
    const tabs = await chrome.tabs.query({ url: `${url.protocol}//${url.hostname}/*` })

    console.log("[v0] Found tabs:", tabs.length)

    for (const tab of tabs) {
      // Inject content script if needed and fill the OTP
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: (code) => {
          // This function runs in the page context
          const inputs = document.querySelectorAll(
            'input[type="text"], input[type="password"], input[type="tel"], input[type="number"]',
          )

          for (const input of inputs) {
            // Look for OTP-related inputs
            const name = (input.name || "").toLowerCase()
            const id = (input.id || "").toLowerCase()
            const placeholder = (input.placeholder || "").toLowerCase()
            const ariaLabel = (input.getAttribute("aria-label") || "").toLowerCase()

            const otpKeywords = ["otp", "code", "verification", "קוד", "אימות", "סיסמה", "password"]
            const isOtpField = otpKeywords.some(
              (keyword) =>
                name.includes(keyword) ||
                id.includes(keyword) ||
                placeholder.includes(keyword) ||
                ariaLabel.includes(keyword),
            )

            if (isOtpField && !input.value) {
              input.value = code
              input.dispatchEvent(new Event("input", { bubbles: true }))
              input.dispatchEvent(new Event("change", { bubbles: true }))
              console.log("[v0] OTP filled in field:", input)
              break
            }
          }
        },
        args: [otpCode],
      })
    }
  } catch (error) {
    console.error("[v0] Error filling OTP:", error)
  }
}
