using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace NokiaIBConverter
{
    // Best-effort IB writer that implements IWriter interface.
    // It writes a header of 0x244 bytes and fixed-size blocks of 0x250 bytes per contact.
    public class IbWriter : IWriter, IDisposable
    {
        private const int HeaderOffset = 0x244;
        private const int SectionOffset = 0x250;
        private FileStream _fs;
        private BinaryWriter _bw;

        public IbWriter(string outPath)
        {
            _fs = new FileStream(outPath, FileMode.Create, FileAccess.Write);
            _bw = new BinaryWriter(_fs);
            // write header placeholder
            _bw.Write(new byte[HeaderOffset]);
        }

        // IWriter.Write will be called per ContactEntry by Converter
        public void Write(ContactEntry entry)
        {
            var block = new byte[SectionOffset];
            // marker similar to original: first two bytes set to {56, 2}
            block[0] = 56;
            block[1] = 2;

            // name into offset 2..65 (64 bytes)
            var name = (entry.FirstName + " " + entry.LastName).Trim();
            var nameBytes = Encoding.UTF8.GetBytes(name);
            int nameLen = Math.Min(64, nameBytes.Length);
            Array.Copy(nameBytes, 0, block, 2, nameLen);

            // phone numbers: put up to 3 numbers, each 32 bytes
            int phoneOffset = 2 + 64;
            if (entry.PhoneNumber != null && entry.PhoneNumber.Length>0)
            {
                var pb = Encoding.UTF8.GetBytes(entry.PhoneNumber[0]);
                Array.Copy(pb, 0, block, phoneOffset + 0*32, Math.Min(32, pb.Length));
            }
            if (entry.PhoneNumber2 != null && entry.PhoneNumber2.Length>0)
            {
                var pb = Encoding.UTF8.GetBytes(entry.PhoneNumber2[0]);
                Array.Copy(pb, 0, block, phoneOffset + 1*32, Math.Min(32, pb.Length));
            }
            if (entry.PhoneNumber3 != null && entry.PhoneNumber3.Length>0)
            {
                var pb = Encoding.UTF8.GetBytes(entry.PhoneNumber3[0]);
                Array.Copy(pb, 0, block, phoneOffset + 2*32, Math.Min(32, pb.Length));
            }

            _bw.Write(block);
        }

        public void Dispose()
        {
            _bw?.Flush();
            _bw?.Dispose();
            _fs?.Dispose();
        }
    }
}
