package com.yossi.kosherappstore

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import java.io.File
import java.io.DataOutputStream
import java.io.FileOutputStream
import java.io.IOException

object SharedPreferencesHelper {
    private const val TAG = "SharedPreferencesHelper"
    private const val CUSTOM_PREFS_FILENAME = "KosherAppStorePrefs.json"
    private const val FALLBACK_PREFS_DIR = "prefs_fallback"

    private var preferencesMap: MutableMap<String, Any?> = mutableMapOf()
    private var actualPrefsFile: File? = null
    private var initializedContext: Context? = null
    private val gson = Gson()
    private var isInitialized = false

    fun init(context: Context) {
        if (isInitialized) return
        initializedContext = context.applicationContext

        // החלטה על מיקום קובץ ה-SharedPreferences
        // כרגע, נשאיר אותו במיקום ברירת המחדל של האפליקציה (בתוך filesDir/prefs_fallback)
        // כדי למנוע סיבוכים עם כתיבה ל-/system עבור קובץ ההגדרות הראשי.
        // קובץ הקונפיגורציה ב-/system ישמור רק את נתיב האחסון של האפליקציות.
        val fallbackDir = File(initializedContext!!.filesDir, FALLBACK_PREFS_DIR)
        if (!fallbackDir.exists()) fallbackDir.mkdirs()
        actualPrefsFile = File(fallbackDir, CUSTOM_PREFS_FILENAME)

        Log.d(TAG, "SharedPreferences will be managed at: ${actualPrefsFile?.absolutePath}")
        loadPrefsFromFile()
        isInitialized = true
    }

    private fun loadPrefsFromFile() {
        actualPrefsFile?.let { file ->
            if (file.exists()) {
                try {
                    val content = file.readText() // קריאה רגילה
                    if (content.isNotEmpty()) {
                        val type = object : TypeToken<MutableMap<String, Any?>>() {}.type
                        preferencesMap = gson.fromJson(content, type) ?: mutableMapOf()
                        Log.d(TAG, "Preferences loaded from ${file.absolutePath}. Count: ${preferencesMap.size}")
                    } else {
                        preferencesMap.clear()
                        Log.d(TAG, "Preferences file is empty: ${file.absolutePath}")
                    }
                } catch (e: Exception) {
                    Log.e(TAG, "Error parsing/loading preferences from ${file.absolutePath}", e)
                    preferencesMap.clear()
                }
            } else {
                Log.w(TAG, "Preferences file does not exist: ${file.absolutePath}. Starting with empty map.")
                preferencesMap.clear()
            }
        }
    }

    private fun savePrefsToFile() {
        actualPrefsFile?.let { file ->
            val json = gson.toJson(preferencesMap)
            try {
                if (!file.parentFile!!.exists()) {
                    file.parentFile!!.mkdirs()
                }
                file.writeText(json) // כתיבה רגילה
                Log.d(TAG, "Preferences saved to ${file.absolutePath}")
            } catch (e: Exception) {
                Log.e(TAG, "Error saving preferences to ${file.absolutePath}", e)
            }
        }
    }

    fun getString(key: String, defaultValue: String?): String? = preferencesMap[key] as? String ?: defaultValue
    fun putString(key: String, value: String?) { if (value == null) preferencesMap.remove(key) else preferencesMap[key] = value; savePrefsToFile() }
    fun getBoolean(key: String, defaultValue: Boolean): Boolean = preferencesMap[key] as? Boolean ?: defaultValue
    fun putBoolean(key: String, value: Boolean) { preferencesMap[key] = value; savePrefsToFile() }
    fun contains(key: String): Boolean = preferencesMap.containsKey(key)

    // פונקציות לקריאה/כתיבה של קבצים אחרים עם רוט (לא קובץ ה-SharedPreferences עצמו)
    // הפכו ל-internal כדי ש-MainActivity יוכל לקרוא את קובץ הקונפיגורציה
    internal fun readFileWithRootBlocking(filePath: String) : String? {
        var content: String? = null; var process: Process? = null
        try {
            process = Runtime.getRuntime().exec("su")
            DataOutputStream(process.outputStream).use { os -> os.writeBytes("cat '$filePath'\n"); os.writeBytes("exit\n"); os.flush() }
            content = process.inputStream.bufferedReader().readText().trim()
            if (process.waitFor() != 0 || content.isNullOrEmpty()) { val err = process.errorStream.bufferedReader().readText(); Log.e(TAG, "Fail read '$filePath' (root): $err"); return null }
        } catch (e: Exception) { Log.e(TAG, "Exc read '$filePath' (root)", e); return null }
        finally { process?.destroy() }
        return content
    }

    // writeFileWithRootBlocking כרגע לא בשימוש ישיר מחוץ ל-Helper אם ה-SharedPreferences עצמו לא נשמר ברוט
    // אם נצטרך אותו, נהפוך גם אותו ל-internal.

    inline fun edit(action: SharedPreferencesEditorHelper.() -> Unit) { SharedPreferencesEditorHelper(this).apply(action).applyChanges() }
    class SharedPreferencesEditorHelper(private val helper: SharedPreferencesHelper) {
        private val tempMap = helper.preferencesMap.toMutableMap()
        fun putString(key: String, value: String?): SharedPreferencesEditorHelper { if (value == null) tempMap.remove(key) else tempMap[key] = value; return this }
        fun putBoolean(key: String, value: Boolean): SharedPreferencesEditorHelper { tempMap[key] = value; return this }
        fun remove(key: String): SharedPreferencesEditorHelper { tempMap.remove(key); return this }
        fun clear(): SharedPreferencesEditorHelper { tempMap.clear(); return this }
        fun applyChanges() { helper.preferencesMap = tempMap; helper.savePrefsToFile() }
    }
}