package com.yossi.kosherappstore

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import androidx.appcompat.app.AppCompatActivity
import android.os.Bundle
import android.util.Log
import android.view.MenuItem // הוספנו עבור כפתור "אחורה"
import android.view.View
import android.widget.TextView
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.core.content.edit
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import java.io.File
import java.lang.reflect.Type

class ManageStoreAppsActivity : AppCompatActivity() {

    private lateinit var recyclerView: RecyclerView
    private lateinit var adapter: ManageAppsAdapter
    private lateinit var appList: MutableList<AppInfo>
    private lateinit var sharedPreferences: SharedPreferences
    private val gson = Gson()
    private lateinit var textViewNoApps: TextView
    private var dataChanged = false // דגל לדעת אם בוצע שינוי

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_manage_store_apps)

        supportActionBar?.setDisplayHomeAsUpEnabled(true) // הוספת כפתור "אחורה" ב-ActionBar
        title = "ניהול אפליקציות בחנות"

        sharedPreferences = getSharedPreferences(MainActivity.PREFS_NAME, Context.MODE_PRIVATE)
        appList = loadAppsListFromPrefs().toMutableList()

        recyclerView = findViewById(R.id.recyclerViewManageApps)
        textViewNoApps = findViewById(R.id.textViewNoAppsToManage)
        recyclerView.layoutManager = LinearLayoutManager(this)

        adapter = ManageAppsAdapter(this, appList.toMutableList()) { appInfo, position -> // שלח עותק של הרשימה
            showDeleteConfirmationDialog(appInfo, position)
        }
        recyclerView.adapter = adapter

        updateEmptyView()
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        // טיפול בלחיצה על כפתור "אחורה" ב-ActionBar
        if (item.itemId == android.R.id.home) {
            onBackPressedDispatcher.onBackPressed()
            return true
        }
        return super.onOptionsItemSelected(item)
    }


    private fun updateEmptyView() {
        if (adapter.appList.isEmpty()) { // בדוק את הרשימה של ה-Adapter
            recyclerView.visibility = View.GONE
            textViewNoApps.visibility = View.VISIBLE
        } else {
            recyclerView.visibility = View.VISIBLE
            textViewNoApps.visibility = View.GONE
        }
    }

    private fun showDeleteConfirmationDialog(appInfo: AppInfo, position: Int) {
        AlertDialog.Builder(this)
            .setTitle("מחק מהחנות")
            .setMessage("האם אתה בטוח שברצונך למחוק את '${appInfo.name}' (${appInfo.packageName}) מהחנות לצמיתות?\nפעולה זו תמחק גם את קובץ ה-APK והתמונה המשויכים.")
            .setPositiveButton("מחק") { _, _ ->
                performDeleteFromStore(appInfo, position)
            }
            .setNegativeButton("ביטול", null)
            .show()
    }

    private fun performDeleteFromStore(appInfo: AppInfo, position: Int) {
        val originalApkFileName = appInfo.apkFileName
        val originalImageFileName = appInfo.imageFileName

        // 1. הסר מהרשימה ב-Adapter (זה יסיר מהעותק שה-Adapter מחזיק)
        adapter.removeItem(position)

        // 2. שמור את הרשימה המעודכנת (מה-Adapter) ל-SharedPreferences
        saveAppsListToPrefs(adapter.appList)

        // 3. מחק קבצים רק אחרי שהשמירה הצליחה והפריט הוסר מהרשימה
        if (originalApkFileName.isNotEmpty()) {
            val apkFile = File(filesDir, originalApkFileName)
            if (apkFile.exists() && apkFile.delete()) {
                Log.d("ManageStoreApps", "APK $originalApkFileName deleted.")
            } else {
                Log.w("ManageStoreApps", "Failed to delete APK $originalApkFileName or it didn't exist.")
            }
        }
        originalImageFileName?.let {
            if (it.isNotEmpty()) {
                val imageFile = File(filesDir, it)
                if (imageFile.exists() && imageFile.delete()) {
                    Log.d("ManageStoreApps", "Image $it deleted.")
                } else {
                    Log.w("ManageStoreApps", "Failed to delete image $it or it didn't exist.")
                }
            }
        }

        Toast.makeText(this, "'${appInfo.name}' נמחקה מהחנות.", Toast.LENGTH_SHORT).show()
        updateEmptyView()
        dataChanged = true // סמן שבוצע שינוי
    }


    private fun loadAppsListFromPrefs(): List<AppInfo> {
        val jsonApps = sharedPreferences.getString(MainActivity.KEY_APPS_LIST, null)
        return if (jsonApps != null) {
            val type: Type = object : TypeToken<ArrayList<AppInfo>>() {}.type
            gson.fromJson(jsonApps, type) ?: ArrayList()
        } else {
            ArrayList()
        }
    }

    private fun saveAppsListToPrefs(updatedAppList: List<AppInfo>) {
        val jsonApps = gson.toJson(updatedAppList)
        sharedPreferences.edit {
            putString(MainActivity.KEY_APPS_LIST, jsonApps)
        }
        Log.d("ManageStoreApps", "App list saved to prefs. Count: ${updatedAppList.size}")
    }

    // החזרת תוצאה ל-SettingsActivity אם בוצע שינוי
    override fun finish() {
        if (dataChanged) {
            setResult(Activity.RESULT_OK)
        } else {
            setResult(Activity.RESULT_CANCELED)
        }
        super.finish()
    }
}