package com.yossi.kosherappstore

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.text.InputType
import android.util.Log
import android.view.LayoutInflater
import android.view.Menu
import android.view.MenuItem
import android.widget.EditText
import android.widget.ProgressBar
import android.widget.TextView
import android.widget.Toast
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.FileProvider
import androidx.core.net.toUri
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import kotlinx.coroutines.*
import java.io.BufferedInputStream
import java.io.DataOutputStream
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.lang.reflect.Type
import java.net.URL

class MainActivity : AppCompatActivity(),
    AppAdapter.OnInstallClickListener,
    AppAdapter.OnUninstallClickListener {

    companion object {
        private const val TAG = "MainActivity"
        const val PREFS_NAME = "KosherAppStorePrefs" // Will be used by SharedPreferencesHelper
        const val KEY_APPS_LIST = "appsList"
        const val KEY_SETTINGS_PASSWORD = "settingsPassword"
        const val DEFAULT_PASSWORD = "yossi6547"
        const val KEY_INSTALL_LOCATION = "installLocation"
        const val KEY_USE_ROOT_INSTALL = "useRootInstall"
        const val KEY_NORMAL_INSTALL_MODE = "normalInstallMode"
        const val KEY_ROOT_INSTALL_METHOD = "rootInstallMethod"
        const val ROOT_METHOD_PM_INSTALL = "pm_install"
        const val ROOT_METHOD_COPY_TO_SYSTEM = "copy_to_system"
        const val KEY_ENABLE_SEND_FEEDBACK = "enableSendFeedback"
        const val KEY_ABOUT_TEXT = "aboutText"
        const val DEFAULT_ABOUT_TEXT = "זוהי חנות אפליקציות כשרה.\n\nתוכל להוסיף לכאן קישורים ומידע נוסף."
        const val FEEDBACK_EMAIL = "c51525355@gmail.com"
        const val KEY_USE_CUSTOM_STORAGE_PATH = "useCustomStoragePath"
        const val KEY_CUSTOM_STORAGE_PATH = "customStoragePath"
        const val KOSHER_STORE_BASE_DIR_SYSTEM = "/system/kosherstore"
        const val CONFIG_FILE_NAME = "storepath.cfg"
        const val APPS_SUBDIR_NAME_IN_CONFIG_PATH = "apps"
        const val DEFAULT_SYSTEM_STORAGE_PATH_FOR_APPS = "$KOSHER_STORE_BASE_DIR_SYSTEM/$APPS_SUBDIR_NAME_IN_CONFIG_PATH"

        fun getActiveStoreFilesBasePath(context: Context): File {
            SharedPreferencesHelper.init(context.applicationContext)
            val useCustomPathEnabledInSettings = SharedPreferencesHelper.getBoolean(KEY_USE_CUSTOM_STORAGE_PATH, false)

            if (useCustomPathEnabledInSettings) {
                val pathFromSystemConfigFile = readPathFromSystemConfigFileWithRootBlocking(context)
                if (!pathFromSystemConfigFile.isNullOrEmpty()) {
                    Log.d(TAG, "Using configured path from system file: $pathFromSystemConfigFile")
                    return File(pathFromSystemConfigFile)
                } else {
                    val userEnteredPathInSettings = SharedPreferencesHelper.getString(KEY_CUSTOM_STORAGE_PATH, "")?.trim()
                    if(!userEnteredPathInSettings.isNullOrEmpty()){
                        return if(userEnteredPathInSettings.startsWith("/system")){
                            val effectivePath = File(userEnteredPathInSettings, APPS_SUBDIR_NAME_IN_CONFIG_PATH)
                            Log.w(TAG, "System config file empty/unreadable. Using user-provided system path from Prefs. Effective path for APPS: ${effectivePath.absolutePath}")
                            effectivePath
                        } else {
                            Log.w(TAG, "System config file empty/unreadable. Using user-provided non-system path from Prefs: $userEnteredPathInSettings")
                            File(userEnteredPathInSettings)
                        }
                    } else {
                        Log.w(TAG, "Custom path ON, but no path in Prefs and no system config. Defaulting APPS to: $DEFAULT_SYSTEM_STORAGE_PATH_FOR_APPS")
                        return File(DEFAULT_SYSTEM_STORAGE_PATH_FOR_APPS)
                    }
                }
            } else {
                Log.d(TAG, "Using internal app storage (filesDir) for APPS: ${context.filesDir.absolutePath}")
                return context.filesDir
            }
        }

        private fun readPathFromSystemConfigFileWithRootBlocking(context: Context): String? {
            val configFilePath = "$KOSHER_STORE_BASE_DIR_SYSTEM/$CONFIG_FILE_NAME"
            // SharedPreferencesHelper.init(context.applicationContext) // Already called in getActiveStoreFilesBasePath
            return SharedPreferencesHelper.readFileWithRootBlocking(configFilePath)
        }
    }

    private lateinit var recyclerViewApps: RecyclerView
    private lateinit var appAdapter: AppAdapter
    private val gson = Gson()
    private val activityScope = CoroutineScope(Dispatchers.Main + SupervisorJob())
    private var progressDialog: AlertDialog? = null
    private var downloadJob: Job? = null

    private val settingsActivityResultLauncher =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            if (result.resultCode == Activity.RESULT_OK) {
                Log.d(TAG, "Returned from Settings with RESULT_OK, reloading apps and adapter path.")
                if (::appAdapter.isInitialized) {
                    appAdapter.updateBaseStoragePath(getActiveStoreFilesBasePath(this))
                }
                loadAndDisplayApps()
                invalidateOptionsMenu()
            }
        }

    private var appToUninstallPackageName: String? = null
    private val uninstallActivityResultLauncher =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
            appToUninstallPackageName?.let { packageName ->
                try { packageManager.getPackageInfo(packageName, 0); Toast.makeText(this, "הסרת $packageName בוטלה/נכשלה", Toast.LENGTH_SHORT).show()
                } catch (e: PackageManager.NameNotFoundException) { Toast.makeText(this, "$packageName הוסר", Toast.LENGTH_SHORT).show() }
            }
            appToUninstallPackageName = null; loadAndDisplayApps()
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        SharedPreferencesHelper.init(applicationContext) // אתחול ראשוני של ה-Helper
        setContentView(R.layout.activity_main)
        title = getString(R.string.app_name)

        if (!SharedPreferencesHelper.contains(KEY_SETTINGS_PASSWORD)) SharedPreferencesHelper.edit { putString(KEY_SETTINGS_PASSWORD, DEFAULT_PASSWORD) }
        if (!SharedPreferencesHelper.contains(KEY_ROOT_INSTALL_METHOD)) SharedPreferencesHelper.edit { putString(KEY_ROOT_INSTALL_METHOD, ROOT_METHOD_PM_INSTALL) }
        if (!SharedPreferencesHelper.contains(KEY_ABOUT_TEXT)) SharedPreferencesHelper.edit { putString(KEY_ABOUT_TEXT, getString(R.string.default_about_text)) }
        if (!SharedPreferencesHelper.contains(KEY_ENABLE_SEND_FEEDBACK)) SharedPreferencesHelper.edit { putBoolean(KEY_ENABLE_SEND_FEEDBACK, true) }
        if (!SharedPreferencesHelper.contains(KEY_NORMAL_INSTALL_MODE)) SharedPreferencesHelper.edit { putBoolean(KEY_NORMAL_INSTALL_MODE, false) }
        if (!SharedPreferencesHelper.contains(KEY_USE_ROOT_INSTALL)) SharedPreferencesHelper.edit { putBoolean(KEY_USE_ROOT_INSTALL, true) }
        if (!SharedPreferencesHelper.contains(KEY_INSTALL_LOCATION)) SharedPreferencesHelper.edit { putString(KEY_INSTALL_LOCATION, "system/app") }
        if (!SharedPreferencesHelper.contains(KEY_USE_CUSTOM_STORAGE_PATH)) SharedPreferencesHelper.edit { putBoolean(KEY_USE_CUSTOM_STORAGE_PATH, false) }
        if (!SharedPreferencesHelper.contains(KEY_CUSTOM_STORAGE_PATH)) SharedPreferencesHelper.edit { putString(KEY_CUSTOM_STORAGE_PATH, "") }

        recyclerViewApps = findViewById(R.id.recyclerViewApps)
        recyclerViewApps.layoutManager = LinearLayoutManager(this)
        appAdapter = AppAdapter(this, getActiveStoreFilesBasePath(this), this, this)
        recyclerViewApps.adapter = appAdapter

        Log.d(TAG, "onCreate: Initial base path for adapter: ${getActiveStoreFilesBasePath(this).absolutePath}")
        loadAndDisplayApps()
    }

    override fun onResume() { super.onResume(); if(::appAdapter.isInitialized) appAdapter.updateBaseStoragePath(getActiveStoreFilesBasePath(this)); loadAndDisplayApps(); invalidateOptionsMenu() }
    override fun onDestroy() { super.onDestroy(); downloadJob?.cancel(); activityScope.cancel(); progressDialog?.dismiss() }
    override fun onCreateOptionsMenu(menu: Menu): Boolean { menuInflater.inflate(R.menu.main_menu, menu); return true }
    override fun onPrepareOptionsMenu(menu: Menu?): Boolean { menu?.findItem(R.id.action_send_feedback)?.isVisible = SharedPreferencesHelper.getBoolean(KEY_ENABLE_SEND_FEEDBACK, true); return super.onPrepareOptionsMenu(menu) }
    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return when (item.itemId) {
            // R.id.action_settings -> הוסר, מכיוון שפריט זה לא קיים בתפריט של הגרסה הנעולה
            R.id.action_send_feedback -> {
                sendFeedbackEmail()
                true
            }
            R.id.action_about -> {
                startActivity(Intent(this, AboutActivity::class.java))
                true
            }
            else -> super.onOptionsItemSelected(item)
        }
    }
    private fun showPasswordDialogForSettings() {
        val builder = AlertDialog.Builder(this); builder.setTitle("הזן סיסמה להגדרות החנות"); val input = EditText(this).apply { inputType = InputType.TYPE_CLASS_TEXT or InputType.TYPE_TEXT_VARIATION_PASSWORD }; builder.setView(input)
        builder.setPositiveButton("אישור") { dialog, _ -> val p = input.text.toString(); if (p == SharedPreferencesHelper.getString(KEY_SETTINGS_PASSWORD, DEFAULT_PASSWORD)) settingsActivityResultLauncher.launch(Intent(this, SettingsActivity::class.java)) else Toast.makeText(this, "סיסמה שגויה", Toast.LENGTH_SHORT).show(); dialog.dismiss() }
        builder.setNegativeButton(getString(R.string.cancel_button_text)) { dialog, _ -> dialog.cancel() }; builder.show()
    }

    private fun sendFeedbackEmail() {
        val s = "${getString(R.string.feedback_email_subject_prefix)} (${getString(R.string.app_name)})"; val intent = Intent(Intent.ACTION_SENDTO).apply { data="mailto:".toUri(); putExtra(Intent.EXTRA_EMAIL, arrayOf(FEEDBACK_EMAIL)); putExtra(Intent.EXTRA_SUBJECT, s) }
        try { startActivity(Intent.createChooser(intent, getString(R.string.feedback_email_chooser_title))); (getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager).setPrimaryClip(ClipData.newPlainText("Email", FEEDBACK_EMAIL)); Toast.makeText(this, "${getString(R.string.feedback_email_copied_toast)}: $FEEDBACK_EMAIL", Toast.LENGTH_LONG).show() }
        catch (ex: ActivityNotFoundException) { Toast.makeText(this, getString(R.string.feedback_no_email_app_toast), Toast.LENGTH_SHORT).show() }
    }

    private fun loadAndDisplayApps() {
        Log.d(TAG, "loadAndDisplayApps: Loading apps."); val apps = loadAppsListFromPrefs(); Log.d(TAG, "loadAndDisplayApps: Submitting ${apps.size} apps.")
        if(::appAdapter.isInitialized) appAdapter.submitList(apps.toMutableList()) else { Log.e(TAG, "Adapter not init in loadAndDisplay!"); appAdapter = AppAdapter(this, getActiveStoreFilesBasePath(this), this, this); recyclerViewApps.adapter = appAdapter; appAdapter.submitList(apps.toMutableList()) }
    }

    private fun loadAppsListFromPrefs(): List<AppInfo> {
        val json = SharedPreferencesHelper.getString(KEY_APPS_LIST, null); return if (json != null) { val type: Type = object:TypeToken<ArrayList<AppInfo>>(){}.type; try { gson.fromJson(json, type) ?: ArrayList() } catch(e:Exception){ Log.e(TAG, "Parse error", e); ArrayList()}} else ArrayList()
    }

    override fun onInstallClick(appInfo: AppInfo) {
        val baseDir = getActiveStoreFilesBasePath(this); Log.d(TAG, "onInstallClick for: ${appInfo.name}, URL: ${appInfo.apkUrl}, BaseDir: ${baseDir.absolutePath}")
        if(appInfo.apkFileName.isBlank()){ Toast.makeText(this, "שם קובץ APK חסר עבור '${appInfo.name}'.", Toast.LENGTH_LONG).show(); return }
        if (!appInfo.apkUrl.isNullOrBlank()) downloadAndInstallApk(appInfo, baseDir)
        else { val apkFile = File(baseDir, appInfo.apkFileName); if (apkFile.exists()) proceedWithInstallation(apkFile, appInfo) else Toast.makeText(this, "APK '${appInfo.apkFileName}' לא קיים ב-${baseDir.absolutePath} ואין URL.", Toast.LENGTH_LONG).show() }
    }

    private fun downloadAndInstallApk(appInfo: AppInfo, baseDir: File) {
        val destinationFile = File(baseDir, appInfo.apkFileName); val dialogView = LayoutInflater.from(this).inflate(R.layout.dialog_progress, null); val progressBar = dialogView.findViewById<ProgressBar>(R.id.progressBarDownload); val progressText = dialogView.findViewById<TextView>(R.id.textViewProgress)
        progressDialog?.dismiss(); progressDialog = AlertDialog.Builder(this).setTitle("${getString(R.string.download_dialog_title_prefix)} ${appInfo.name}").setView(dialogView).setCancelable(false).setNegativeButton(getString(R.string.download_dialog_cancel_button)) { _, _ -> downloadJob?.cancel("User cancelled"); Toast.makeText(this, getString(R.string.download_cancelled_toast), Toast.LENGTH_SHORT).show() }.create(); progressDialog?.show()
        progressBar.isIndeterminate = true; progressText.text = getString(R.string.download_dialog_connecting)
        downloadJob = activityScope.launch(Dispatchers.IO) {
            var success = false; var errMsg: String? = "שגיאה לא ידועה"
            try {
                val url = URL(appInfo.apkUrl); val conn = url.openConnection().apply{connectTimeout=15000;readTimeout=30000}; conn.connect()
                val totalSize: Long = if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) conn.contentLengthLong else {val l=conn.contentLength; if(l == -1) -1L else l.toLong()}
                withContext(Dispatchers.Main) { if(progressDialog?.isShowing!=true) return@withContext; if(totalSize>0){progressBar.isIndeterminate=false; progressBar.max=100; progressText.text=getString(R.string.download_progress_percent,0)} else {progressBar.isIndeterminate=true; progressText.text=getString(R.string.download_dialog_downloading_unknown_size)}}
                BufferedInputStream(conn.inputStream).use { input -> FileOutputStream(destinationFile).use { output -> val data = ByteArray(8192); var totalRead: Long = 0; var readNow: Int
                    while (input.read(data).also { readNow = it } != -1 && isActive) { totalRead += readNow.toLong(); output.write(data, 0, readNow)
                        if (totalSize > 0) { val p = ((totalRead * 100) / totalSize).toInt(); withContext(Dispatchers.Main) { if(progressDialog?.isShowing==true){progressBar.progress = p; progressText.text = getString(R.string.download_progress_percent, p)}} }
                        else { withContext(Dispatchers.Main) { if(progressDialog?.isShowing==true) progressText.text = getString(R.string.download_progress_kb, totalRead / 1024)} }
                    } } }
                if(isActive) success = true else destinationFile.delete()
            } catch (e: CancellationException) { errMsg = getString(R.string.download_cancelled_toast); Log.d(TAG, "Download cancelled: ${appInfo.apkUrl}", e)
            } catch (e: IOException) { errMsg = "שגיאת רשת: ${e.localizedMessage}"; Log.e(TAG, "IOException for ${appInfo.apkUrl}", e)
            } catch (e: Exception) { errMsg = e.localizedMessage ?: "שגיאה כללית"; Log.e(TAG, "General error for ${appInfo.apkUrl}", e) }
            finally { if (!success && destinationFile.exists()) destinationFile.delete() }
            withContext(Dispatchers.Main) { progressDialog?.dismiss(); progressDialog = null
                if(success) proceedWithInstallation(destinationFile, appInfo)
                else if(downloadJob?.isCancelled == false && isActive) Toast.makeText(applicationContext, getString(R.string.download_error_toast, errMsg), Toast.LENGTH_LONG).show()
            }
        }
    }

    private fun proceedWithInstallation(apkFileToInstall: File, appInfo: AppInfo) {
        Toast.makeText(this, "ממשיך להתקנה של: ${appInfo.name}", Toast.LENGTH_SHORT).show()
        val useNormalInstall = SharedPreferencesHelper.getBoolean(KEY_NORMAL_INSTALL_MODE, false)
        val useRootGeneral = SharedPreferencesHelper.getBoolean(KEY_USE_ROOT_INSTALL, true)
        if (useNormalInstall) { installApkNormally(apkFileToInstall) }
        else if (useRootGeneral) {
            val rootMethod = SharedPreferencesHelper.getString(KEY_ROOT_INSTALL_METHOD, ROOT_METHOD_PM_INSTALL)
            if (rootMethod == ROOT_METHOD_PM_INSTALL) executePmInstallRootCommand(apkFileToInstall.absolutePath)
            else { val pathPref = SharedPreferencesHelper.getString(KEY_INSTALL_LOCATION, "system/app"); val dirName = appInfo.packageName; if (dirName.isBlank()) { Toast.makeText(this, "שם חבילה חסר", Toast.LENGTH_LONG).show(); return }; val targetPath = "/$pathPref/$dirName"; executeCopyToSystemRootCommands(apkFileToInstall.absolutePath, targetPath, appInfo.apkFileName) }
        } else Toast.makeText(this, "אף שיטת התקנה לא הופעלה.", Toast.LENGTH_LONG).show()
    }

    override fun onUninstallClick(appInfo: AppInfo) {
        AlertDialog.Builder(this).setTitle("הסר התקנה").setMessage("האם אתה בטוח שברצונך להסיר את ההתקנה של '${appInfo.name}' מהמכשיר?").setPositiveButton(getString(R.string.uninstall_button_text)) { dialog, _ -> performUninstallAppFromDevice(appInfo); dialog.dismiss() }.setNegativeButton(getString(R.string.cancel_button_text), null).show()
    }

    private fun performUninstallAppFromDevice(appInfo: AppInfo) {
        try { packageManager.getPackageInfo(appInfo.packageName, 0) } catch (e: PackageManager.NameNotFoundException) { Toast.makeText(this, "'${appInfo.name}' כבר לא מותקנת.", Toast.LENGTH_SHORT).show(); loadAndDisplayApps(); return }
        val wasSystem = !SharedPreferencesHelper.getBoolean(KEY_NORMAL_INSTALL_MODE, false) && SharedPreferencesHelper.getBoolean(KEY_USE_ROOT_INSTALL, true) && SharedPreferencesHelper.getString(KEY_ROOT_INSTALL_METHOD, ROOT_METHOD_PM_INSTALL) == ROOT_METHOD_COPY_TO_SYSTEM
        if (wasSystem) { val pathPref = SharedPreferencesHelper.getString(KEY_INSTALL_LOCATION, "system/app"); val dir = "/$pathPref/${appInfo.packageName}"; executeRemoveSystemAppRootCommands(dir, appInfo.packageName) }
        else { appToUninstallPackageName = appInfo.packageName; val intent = Intent(Intent.ACTION_DELETE).apply { data = "package:${appInfo.packageName}".toUri(); putExtra(Intent.EXTRA_RETURN_RESULT, true) }; try { uninstallActivityResultLauncher.launch(intent) } catch (e: Exception) { Toast.makeText(this, "שגיאה בהסרה: ${e.message}",Toast.LENGTH_LONG).show(); appToUninstallPackageName=null } }
    }

    private fun installApkNormally(apkFile: File) {
        try { val uri = FileProvider.getUriForFile(this, "${applicationContext.packageName}.provider", apkFile); val intent = Intent(Intent.ACTION_VIEW).apply { setDataAndType(uri, "application/vnd.android.package-archive"); addFlags(Intent.FLAG_ACTIVITY_NEW_TASK); addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION) }; startActivity(intent) }
        catch (e: Exception) { Toast.makeText(this, "שגיאה בהתקנה: ${e.message}", Toast.LENGTH_LONG).show(); Log.e(TAG, "Normal install error", e) }
    }

    private fun executeRootCommandInternal(commands: List<String>, successMsg: String, failureMsgPrefix: String) {
        activityScope.launch(Dispatchers.IO) {
            var success = false; var exitCode = -1; var errOutput = ""
            try {
                val proc = Runtime.getRuntime().exec("su"); DataOutputStream(proc.outputStream).use { os -> val needsMount = commands.any { it.contains("/system/") }; if(needsMount) os.writeBytes("mount -o rw,remount /system\n"); commands.forEach { os.writeBytes("$it\n"); Log.d(TAG,"ROOT_CMD: $it") }; if(needsMount) os.writeBytes("mount -o ro,remount /system\n"); os.writeBytes("exit\n"); os.flush() }
                exitCode = proc.waitFor(); success = exitCode == 0; if (!success) errOutput = proc.errorStream.bufferedReader().readText(); proc.destroy()
            } catch (e: Exception) { errOutput = e.localizedMessage ?: "Unknown Exception"; Log.e(TAG, "Root cmd error", e) }
            withContext(Dispatchers.Main) {
                if (success) Toast.makeText(this@MainActivity, successMsg, Toast.LENGTH_SHORT).show()
                else Toast.makeText(this@MainActivity, "$failureMsgPrefix, קוד: $exitCode, שגיאה: $errOutput", Toast.LENGTH_LONG).show()
                loadAndDisplayApps()
            }
        }
    }
    private fun executeCopyToSystemRootCommands(source: String, targetDir: String, targetName: String) { executeRootCommandInternal(listOf("mkdir -p '$targetDir'", "cp -f '$source' '$targetDir/$targetName'", "chmod 644 '$targetDir/$targetName'", "chown root:root '$targetDir/$targetName'", "chmod 755 '$targetDir'", "chown root:root '$targetDir'"), "הועתק למערכת (ריבוט ייתכן)", "שגיאה בהעתקה למערכת") }
    private fun executePmInstallRootCommand(apkPath: String) { executeRootCommandInternal(listOf("pm install -r -d -g \"$apkPath\""), "הותקן (pm install)", "שגיאה (pm install)") }
    private fun executeRemoveSystemAppRootCommands(targetDir: String, pkgName: String) { executeRootCommandInternal(listOf("rm -rf '$targetDir'", "pm uninstall \"$pkgName\""), "'$pkgName' הוסר מהמערכת (ריבוט ייתכן)", "שגיאה בהסרת '$pkgName'") }
}