package com.yossi.kosherappstore

import android.content.Context
import android.content.pm.PackageManager
import android.graphics.BitmapFactory
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import androidx.core.view.isVisible
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView // ודא שה-import הזה קיים
import java.io.File

class AppAdapter(
    private val context: Context,
    private var baseStoragePath: File, // נתיב בסיס לשליפה מההגדרות
    private val installClickListener: OnInstallClickListener,
    private val uninstallClickListener: OnUninstallClickListener
) : ListAdapter<AppInfo, AppAdapter.AppViewHolder>(AppInfoDiffCallback()) {

    interface OnInstallClickListener {
        fun onInstallClick(appInfo: AppInfo)
    }

    interface OnUninstallClickListener {
        fun onUninstallClick(appInfo: AppInfo)
    }

    fun updateBaseStoragePath(newPath: File) {
        if (baseStoragePath != newPath) {
            baseStoragePath = newPath
            // אם הנתיב השתנה, ייתכן שנצטרך לאלץ רענון של פריטים קיימים
            // notifyDataSetChanged() // אפשרות פשוטה אך פחות יעילה
            // או לנהל מחדש את ה-submitList עם הרשימה המקורית
            // כדי ש-onBindViewHolder יקרא שוב ויטען תמונות מהמקום הנכון.
            // כרגע, נניח ש-submitList מחדש יקרא מ-MainActivity ויספיק.
            Log.d("AppAdapter", "Base storage path updated to: ${newPath.absolutePath}")
        }
    }


    inner class AppViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {
        private val imageViewAppIcon: ImageView = itemView.findViewById(R.id.imageViewAppIcon)
        private val textViewAppName: TextView = itemView.findViewById(R.id.textViewAppName)
        private val textViewAppDescription: TextView = itemView.findViewById(R.id.textViewAppDescription)
        private val buttonInstall: Button = itemView.findViewById(R.id.buttonInstall)
        private val buttonUninstall: Button = itemView.findViewById(R.id.buttonUninstall)

        fun bind(appInfo: AppInfo) {
            Log.d("AppAdapter", "Binding: ${appInfo.name}, Img: ${appInfo.imageFileName}, BasePath for Img: ${baseStoragePath.absolutePath}")
            textViewAppName.text = appInfo.name
            textViewAppDescription.text = appInfo.description
            imageViewAppIcon.contentDescription = context.getString(R.string.app_icon_description_with_name, appInfo.name)

            appInfo.imageFileName?.let { fileName ->
                if (fileName.isNotEmpty()) {
                    val imgFile = File(baseStoragePath, fileName) // שימוש ב-baseStoragePath
                    if (imgFile.exists()) {
                        try {
                            val bitmap = BitmapFactory.decodeFile(imgFile.absolutePath)
                            imageViewAppIcon.setImageBitmap(bitmap)
                        } catch (e: OutOfMemoryError) {
                            Log.e("AppAdapter", "OutOfMemoryError loading image: ${imgFile.absolutePath}", e)
                            imageViewAppIcon.setImageResource(R.mipmap.ic_launcher)
                        } catch (e: Exception) {
                            Log.e("AppAdapter", "Error loading image: ${imgFile.absolutePath}", e)
                            imageViewAppIcon.setImageResource(R.mipmap.ic_launcher)
                        }
                    } else {
                        Log.w("AppAdapter", "Image file not found: ${imgFile.absolutePath}")
                        imageViewAppIcon.setImageResource(R.mipmap.ic_launcher)
                    }
                } else {
                    imageViewAppIcon.setImageResource(R.mipmap.ic_launcher)
                }
            } ?: run {
                imageViewAppIcon.setImageResource(R.mipmap.ic_launcher)
            }

            val appIsInstalled = isPackageInstalled(appInfo.packageName, context.packageManager)
            buttonInstall.isVisible = !appIsInstalled
            buttonUninstall.isVisible = appIsInstalled

            buttonInstall.setOnClickListener { installClickListener.onInstallClick(appInfo) }
            buttonUninstall.setOnClickListener { uninstallClickListener.onUninstallClick(appInfo) }
        }
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): AppViewHolder {
        val view = LayoutInflater.from(context).inflate(R.layout.list_item_app, parent, false)
        return AppViewHolder(view)
    }

    override fun onBindViewHolder(holder: AppViewHolder, position: Int) {
        holder.bind(getItem(position))
    }

    private fun isPackageInstalled(packageName: String, packageManager: PackageManager): Boolean {
        if (packageName.isBlank()) return false
        return try {
            packageManager.getPackageInfo(packageName, 0); true
        } catch (e: PackageManager.NameNotFoundException) { false }
    }
}

class AppInfoDiffCallback : DiffUtil.ItemCallback<AppInfo>() {
    override fun areItemsTheSame(oldItem: AppInfo, newItem: AppInfo): Boolean {
        return oldItem.packageName == newItem.packageName && oldItem.apkFileName == newItem.apkFileName // יותר חזק
    }

    override fun areContentsTheSame(oldItem: AppInfo, newItem: AppInfo): Boolean {
        return oldItem == newItem
    }
}