from flask import Flask, render_template, request, redirect, url_for, session, send_file, flash
import os
import uuid
import zipfile
import json
import requests
import re
from tempfile import mkdtemp
import shutil

app = Flask(__name__)
app.secret_key = 'supersecretkey123'
app.config['UPLOAD_FOLDER'] = mkdtemp()

# Constants
MANIFEST_FILE_NAME = "store_manifest.json"
APKS_DIR_NAME = "apks"
IMAGES_DIR_NAME = "images"

class AppEntry:
    def __init__(self, apk_source=None, is_url=False, **kwargs):
        self.id = kwargs.get('id', str(uuid.uuid4()))
        self.is_url = kwargs.get('is_url', is_url)
        self.apk_source = apk_source or ""
        self.apk_file_name = self._get_filename(is_url)
        self.app_name = kwargs.get('app_name', os.path.splitext(self.apk_file_name)[0])
        self.package_name = self._generate_package_name()
        self.description = kwargs.get('description', f"תיאור עבור {self.app_name}")
        self.image_path = kwargs.get('image_path', "")
        self.image_file_name = kwargs.get('image_file_name', "")
        self.file_size = kwargs.get('file_size', "0 B")
        self.local_path = kwargs.get('local_path', "")

    def _get_filename(self, is_url):
        if is_url:
            try:
                parsed = requests.utils.urlparse(self.apk_source)
                filename = os.path.basename(parsed.path)
                if not filename or filename == '/':
                    filename = f"app_{uuid.uuid4().hex[:6]}.apk"
                return filename
            except:
                return f"app_{uuid.uuid4().hex[:6]}.apk"
        else:
            return os.path.basename(self.apk_source) if self.apk_source else f"local_{uuid.uuid4().hex[:6]}.apk"

    def _generate_package_name(self):
        safe_name = re.sub(r'[^a-z0-9_]', '', self.app_name.lower().replace(" ", "_"))
        return f"com.generated.{safe_name[:30]}.{uuid.uuid4().hex[:6]}"

    def to_dict(self):
        return {
            "name": self.app_name,
            "description": self.description,
            "apkFileName": self.apk_file_name,
            "imageFileName": self.image_file_name if self.image_file_name else None,
            "packageName": self.package_name,
            "apkUrl": self.apk_source if self.is_url else None,
            "fileSize": self._get_file_size_bytes()
        }

    def _get_file_size_bytes(self):
        try:
            size_str = self.file_size.upper()
            if 'KB' in size_str:
                num = float(size_str.replace('KB', '').strip())
                return int(num * 1024)
            elif 'MB' in size_str:
                num = float(size_str.replace('MB', '').strip())
                return int(num * 1024 * 1024)
            elif 'B' in size_str and 'KB' not in size_str and 'MB' not in size_str:
                num = float(size_str.replace('B', '').strip())
                return int(num)
            return 0
        except:
            return 0

@app.route('/')
def index():
    session.setdefault('apk_entries', [])
    session.setdefault('store_settings', {
        'password': 'yossi6547',
        'normalInstallMode': False,
        'useRootInstall': True,
        'rootInstallMethod': 'pm_install',
        'installLocation': 'system/app',
        'enableSendFeedback': True,
        'aboutText': 'זוהי חנות אפליקציות כשרה שנוצרה באמצעות סקריפט הגיבוי.'
    })
    return render_template('index.html',
                         apk_entries=session['apk_entries'],
                         settings=session['store_settings'])

@app.route('/add_local_apk', methods=['POST'])
def add_local_apk():
    if 'apk_file' not in request.files:
        flash('לא נבחר קובץ')
        return redirect(url_for('index'))

    for file in request.files.getlist('apk_file'):
        if file.filename == '' or not file.filename.lower().endswith('.apk'):
            continue

        temp_path = os.path.join(app.config['UPLOAD_FOLDER'], file.filename)
        file.save(temp_path)
        
        entry = AppEntry(apk_source=file.filename)
        entry.local_path = temp_path
        entry.file_size = f"{os.path.getsize(temp_path)/1024:.2f} KB"
        
        session['apk_entries'].append(entry.__dict__)
    
    session.modified = True
    return redirect(url_for('index'))

@app.route('/add_url_apk', methods=['POST'])
def add_url_apk():
    url = request.form.get('apk_url', '').strip()
    if not url.startswith('http'):
        flash('כתובת URL לא תקינה')
        return redirect(url_for('index'))

    try:
        entry = AppEntry(apk_source=url, is_url=True)
        
        # Get file size
        with requests.head(url, allow_redirects=True, timeout=10) as resp:
            if resp.status_code == 200:
                size = int(resp.headers.get('content-length', 0))
                entry.file_size = f"{size/1024:.2f} KB"
        
        session['apk_entries'].append(entry.__dict__)
        session.modified = True
    except Exception as e:
        flash(f'שגיאה בהוספת קובץ: {str(e)}')
    
    return redirect(url_for('index'))

@app.route('/remove_apk/<entry_id>')
def remove_apk(entry_id):
    session['apk_entries'] = [e for e in session['apk_entries'] if e['id'] != entry_id]
    session.modified = True
    return redirect(url_for('index'))

@app.route('/update_settings', methods=['POST'])
def update_settings():
    session['store_settings'] = {
        'password': request.form['password'],
        'normalInstallMode': 'normalInstallMode' in request.form,
        'useRootInstall': 'useRootInstall' in request.form,
        'rootInstallMethod': request.form['rootInstallMethod'],
        'installLocation': request.form['installLocation'],
        'enableSendFeedback': 'enableSendFeedback' in request.form,
        'aboutText': request.form['aboutText']
    }
    session.modified = True
    flash('הגדרות נשמרו בהצלחה!')
    return redirect(url_for('index'))

@app.route('/create_backup')
def create_backup():
    if not session['apk_entries']:
        flash('לא נוספו קבצי APK')
        return redirect(url_for('index'))

    try:
        temp_dir = mkdtemp()
        
        # Create directories
        apks_dir = os.path.join(temp_dir, APKS_DIR_NAME)
        images_dir = os.path.join(temp_dir, IMAGES_DIR_NAME)
        os.makedirs(apks_dir, exist_ok=True)
        os.makedirs(images_dir, exist_ok=True)

        # Prepare manifest
        manifest = {
            "appList": [AppEntry(**e).to_dict() for e in session['apk_entries']],
            **session['store_settings']
        }

        # Process files
        for entry_data in session['apk_entries']:
            entry = AppEntry(**entry_data)
            
            # Handle APK
            dest_apk = os.path.join(apks_dir, entry.apk_file_name)
            if entry.is_url:
                with requests.get(entry.apk_source, stream=True, timeout=30) as r:
                    r.raise_for_status()
                    with open(dest_apk, 'wb') as f:
                        for chunk in r.iter_content(chunk_size=8192):
                            f.write(chunk)
            elif os.path.exists(entry.local_path):
                shutil.copy(entry.local_path, dest_apk)
            
            # Handle image
            if entry.image_path and os.path.exists(entry.image_path):
                dest_img = os.path.join(images_dir, entry.image_file_name)
                shutil.copy(entry.image_path, dest_img)

        # Save manifest
        manifest_path = os.path.join(temp_dir, MANIFEST_FILE_NAME)
        with open(manifest_path, 'w', encoding='utf-8') as f:
            json.dump(manifest, f, indent=4, ensure_ascii=False)

        # Create ZIP
        zip_path = os.path.join(temp_dir, 'backup.zip')
        with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zf:
            for root, _, files in os.walk(temp_dir):
                for file in files:
                    if file == 'backup.zip':
                        continue
                    full_path = os.path.join(root, file)
                    arcname = os.path.relpath(full_path, temp_dir)
                    zf.write(full_path, arcname)
        
        return send_file(zip_path, as_attachment=True, download_name='backup.zip')
    
    except Exception as e:
        flash(f'שגיאה ביצירת הגיבוי: {str(e)}')
        return redirect(url_for('index'))
    finally:
        shutil.rmtree(temp_dir, ignore_errors=True)

if __name__ == '__main__':
    app.run(debug=True)
